/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2021 Red Hat, Inc., and individual contributors
 * as indicated by the @author tags.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jboss.installer.core;

import javax.xml.stream.XMLEventFactory;
import javax.xml.stream.XMLEventReader;

import org.jboss.installer.auto.AutomaticInstallationParsingException;
import org.jboss.installer.postinstall.PostInstallTask;
import org.jboss.installer.screens.ConfigSummaryPanel;

import java.net.URL;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.BiFunction;

public class InstallationData {
    private Path targetFolder;
    private String adminUsername;
    private String password;
    private List<String> excludedPackages = Collections.emptyList();
    private List<String> selectedPackages = Collections.emptyList();
    // use LinkedHashMap to preserve the insert order of values
    private Map<String, ConfigSummaryPanel> summaries = new LinkedHashMap<>();

    private Map<String, URL> mavenRepositories = new HashMap<>();

    private HashMap<Class<? extends PostInstallConfig>, PostInstallConfig> configs = new HashMap<>();

    private List<PostInstallTask> postInstallTasks = new ArrayList<>();

    // not serialized
    private Map<String, URL> defaultMavenRepositories = new HashMap<>();

    public Map<String, URL> getDefaultMavenRepositories() {
        return defaultMavenRepositories;
    }

    public void setDefaultMavenRepositories(Map<String, URL> defaultMavenRepositories) {
        this.defaultMavenRepositories = defaultMavenRepositories;
    }

    public void setTargetFolder(Path targetFolder) {
        this.targetFolder = targetFolder;
    }

    public Path getTargetFolder() {
        return targetFolder;
    }

    public String getAdminUsername() {
        return adminUsername;
    }

    public void setAdminUsername(String adminUsername) {
        this.adminUsername = adminUsername;
    }

    public String getPassword() {
        return password;
    }

    public void setPassword(String password) {
        this.password = password;
    }

    public Map<String, URL> getMavenRepositories() {
        return mavenRepositories;
    }

    public void setMavenRepositories(Map<String, URL> mavenRepositories) {
        this.mavenRepositories = mavenRepositories;
    }

    public void addMavenRepository(String repoId, URL repoUrl) {
        if (!mavenRepositories.containsKey(repoId)) {
            mavenRepositories.put(repoId, repoUrl);
        }
    }

    public List<String> getExcludedPackages() {
        return excludedPackages;
    }

    public void setExcludedPackages(List<String> excludedPackages) {
        this.excludedPackages = excludedPackages;
    }

    public List<String> getSelectedPackages() {
        return selectedPackages;
    }

    public void setSelectedPackages(List<String> selectedPackages) {
        this.selectedPackages = selectedPackages;
    }

    public void addPostInstallTask(PostInstallTask action) {
        if (!postInstallTasks.contains(action)) {
            postInstallTasks.add(action);
        }
    }

    public List<PostInstallTask> getPostInstallTasks() {
        return postInstallTasks;
    }

    public void removePostInstallTask(PostInstallTask action) {
        postInstallTasks.remove(action);
    }

    public <T extends PostInstallConfig> void putConfig(T config) {
        configs.put(config.getClass(), config);
    }

    @SuppressWarnings("unchecked")
    public <T extends PostInstallConfig> T getConfig(Class<? extends PostInstallConfig> klass) {
        return (T) configs.get(klass);
    }

    public Map<Class<? extends PostInstallConfig>, PostInstallConfig> getConfigs() {
        return configs;
    }

    public <T extends PostInstallConfig> void removeConfig(Class<? extends PostInstallConfig> klass) {
        configs.remove(klass);
    }

    public Collection<ConfigSummaryPanel> getSummaries() {
        return Collections.unmodifiableCollection(summaries.values());
    }

    public void removeSummaries(List<String> configurationScreens) {
        configurationScreens.forEach(summaries::remove);
    }

    public void updateSummary(String screenName, String description, HashMap<String, String> attrs) {
        if (summaries.containsKey(screenName)) {
            summaries.get(screenName).updateAttrs(attrs);
        } else {
            ConfigSummaryPanel screenSummary = new ConfigSummaryPanel(description, attrs);
            summaries.put(screenName, screenSummary);
        }
    }

    /**
     * Contains configuration specific to post install task
     */
    public interface PostInstallConfig {
        XMLEventReader serialize(XMLEventFactory eventFactory, Set<String> variables);
        void deserialize(XMLEventReader reader, BiFunction<String, String, String> variableResolver) throws AutomaticInstallationParsingException;
    }
}
