/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2023 Red Hat, Inc., and individual contributors
 * as indicated by the @author tags.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jboss.installer.validators;

import org.jboss.installer.core.LanguageUtils;
import org.jboss.installer.core.ValidationResult;

import static org.jboss.installer.screens.CustomPortConfigurationScreen.DEFAULT_PORT_VALUES;

public class PortOffsetValidator {

    public static final String EMPTY_OFFSET_ERROR_KEY = "port_bindings.empty_offset_value_error";
    public static final String NON_INT_OFFSET_ERROR_KEY = "port_bindings.non_int_offset_value_error";
    public static final String NEGATIVE_OFFSET_ERROR_KEY = "port_bindings.negative_offset_value_error";
    public static final String HIGH_OFFSET_ERROR_KEY = "port_bindings.too_high_offset_value_error";
    public static final int MAX_PORT_VALUE = PortConfigurationValidator.MAX_PORT_VALUE;
    private final LanguageUtils languageUtils;

    public PortOffsetValidator(LanguageUtils languageUtils) {
        this.languageUtils = languageUtils;
    }

    public ValidationResult validate(String offsetText) {
        final int highestPortValue = getMaxPortValue();
        final int maxValue = MAX_PORT_VALUE - getMaxPortValue();

        if (offsetText == null || offsetText.trim().isEmpty()) {
            return ValidationResult.error(languageUtils.getString(EMPTY_OFFSET_ERROR_KEY, maxValue+""));
        }

        final int offset;
        try {
            offset = Integer.parseInt(offsetText);
        } catch (NumberFormatException e) {
            try {
                Long.parseLong(offsetText);
                // it's too long for int but a valid number
                // if it's too long for int it's definitely over our limit
                return ValidationResult.error(languageUtils.getString(HIGH_OFFSET_ERROR_KEY, maxValue+""));
            } catch (NumberFormatException ex) {
                // ok ignore, we're returning error below
            }
            return ValidationResult.error(languageUtils.getString(NON_INT_OFFSET_ERROR_KEY, maxValue+""));
        }

        if (offset < 0) {
            return ValidationResult.error(languageUtils.getString(NEGATIVE_OFFSET_ERROR_KEY, maxValue+""));
        }

        if (offset + highestPortValue > MAX_PORT_VALUE) {
            return ValidationResult.error(languageUtils.getString(HIGH_OFFSET_ERROR_KEY, maxValue+""));
        }

        return ValidationResult.ok();
    }

    int getMaxPortValue() {
        return DEFAULT_PORT_VALUES.values().stream().map(v -> {
                    try {
                        return Integer.parseInt(v);
                    } catch (NumberFormatException e) {
                        return null;
                    }
                })
                .filter(v -> v != null)
                .max(Integer::compare)
                .orElse(0);
    }
}
