/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2021 Red Hat, Inc., and individual contributors
 * as indicated by the @author tags.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jboss.installer.screens;

import org.jboss.installer.common.FileChooserPanel;
import org.jboss.installer.common.InstallerDialogs;
import org.jboss.installer.core.DatabaseDriver;
import org.jboss.installer.core.DownloadUtils;
import org.jboss.installer.core.InstallationData;
import org.jboss.installer.core.LanguageUtils;
import org.jboss.installer.core.Screen;
import org.jboss.installer.core.ValidationResult;
import org.jboss.installer.dialogs.downloader.DownloadDisplayPanel;
import org.jboss.installer.dialogs.downloader.DownloadHandler;
import org.jboss.installer.dialogs.downloader.DownloadManager;
import org.jboss.installer.postinstall.PostInstallTask;
import org.jboss.installer.postinstall.task.JDBCDriverConfig;
import org.jboss.installer.validators.JDBCDriverValidator;
import org.jboss.installer.validators.PathValidator;

import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.filechooser.FileNameExtensionFilter;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.Rectangle;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import static javax.swing.JFileChooser.FILES_ONLY;

public class JDBCDriverScreen extends DefaultScreen{

    public static final String NAME = "JDBCDriverScreen";
    public static final String TITLE_KEY="jdbc_driver.title";
    public static final String DESCRIPTION_KEY = "jdbc_driver.description";
    public static final String VENDOR_KEY = "jdbc_driver.vendor";
    public static final String JAR_LOCATIONS_KEY = "jdbc_driver.jar.locations";
    public static final String JAR_ADD_KEY = "jdbc_driver.jar.add";
    public static final String JAR_REMOVE_KEY = "jdbc_driver.jar.remove";
    public static final String DRIVER_NAME_KEY = "jdbc_driver.name";
    public static final String MODULE_NAME_KEY = "jdbc_driver.module_name";
    public static final String XA_NAME_KEY = "jdbc_driver.xa_name";
    public static final String DIRECTORY_STRUCTURE_KEY = "jdbc_driver.directory_structure";
    private static final int MAX_JARS_COUNT = 9;
    private static final String JDBC_DRIVER_DESCRIPTION_2 = "jdbc_driver.description.2";
    private final JComboBox driverVendors = createDropdown(DatabaseDriver.getDriverVendorNames(), false);
    // create the add/remove buttons first to assign mnemonic first
    private final JButton jarAddButton = createButton(JAR_ADD_KEY, actionEvent -> addJarLocation());
    private final JButton jarRemoveButton = createButton(JAR_REMOVE_KEY, actionEvent -> removeJarLocation());
    private final List<FileChooserPanel> jarLocations = createJarLocationList();
    private final JTextField jdbcNameField = createTextField("");
    private final JTextField moduleNameField = createTextField("");
    private final JTextField xaNameField = createTextField("");
    private final JTextField directoryStructureField = createTextField("");
    private int workingJarLocationIndex = 0;
    private JPanel content;
    private List<String> resolvedJarLocations;
    private DownloadUtils downloadUtils;


    public JDBCDriverScreen(Screen parent, LanguageUtils langUtils, boolean isActive) {
        super(parent, langUtils, isActive);
    }

    @Override
    public String getTitle() {
        return langUtils.getString(TITLE_KEY);
    }

    @Override
    public String getName() {
        return NAME;
    }

    @Override
    public JPanel getContent() {
        content = new JPanel(new GridBagLayout());
        GridBagConstraints c = initializeConstraints();
        content.setBorder(BorderFactory.createEmptyBorder(0,0,0,10));

        c.insets = DESCRIPTION_INSET;
        c.gridwidth = 2;
        content.add(createDescription(DESCRIPTION_KEY), c);
        c.gridy++;
        content.add(createHtmlDescription(JDBC_DRIVER_DESCRIPTION_2), c);
        c.gridy++;

        c.gridwidth = 1;
        addField(content, c, VENDOR_KEY, driverVendors);

        addField(content, c, JAR_LOCATIONS_KEY, jarLocations.get(0));

        addInvisibleJarLocationFileChoosers(content, c);

        addJarLocationButtons(content, c);

        c.gridwidth = 2;
        c.gridx = 0;
        setDriverSettings(getSelectedDatabaseDriver());
        jdbcNameField.setEnabled(false);
        c.gridwidth = 1;
        addField(content, c, DRIVER_NAME_KEY, jdbcNameField);

        moduleNameField.setEnabled(false);
        addField(content, c, MODULE_NAME_KEY, moduleNameField);

        xaNameField.setEnabled(false);
        addField(content, c, XA_NAME_KEY, xaNameField);
        directoryStructureField.setEnabled(false);
        addField(content, c, DIRECTORY_STRUCTURE_KEY, directoryStructureField);
        c.gridy++;
        fillEmptySpace(content, c);

        driverVendors.addActionListener(actionEvent -> {
            setDriverSettings(getSelectedDatabaseDriver());
        });

        return content;
    }

    @Override
    public JComponent getDefaultFocusComponent() {
        return driverVendors;
    }

    private void addJarLocationButtons(JPanel content, GridBagConstraints c) {
        final JPanel buttonPanel = new JPanel(new GridBagLayout());
        final GridBagConstraints c1 = initializeConstraints();
        c1.gridx = 0;
        c1.fill = GridBagConstraints.NONE;
        c1.insets = new Insets(0,0, 0, 5);
        c1.anchor = GridBagConstraints.EAST;
        buttonPanel.add(jarAddButton, c1);
        jarAddButton.setEnabled(canAddMoreJars());

        c1.gridx = 1;
        c1.insets = new Insets(0,5, 0, 0);
        c1.anchor = GridBagConstraints.WEST;
        jarRemoveButton.setEnabled(canRemoveJars());
        buttonPanel.add(jarRemoveButton, c1);

        c.gridwidth = 1;
        c.weightx = 1;
        content.add(buttonPanel, c);
    }

    private void addInvisibleJarLocationFileChoosers(JPanel content, GridBagConstraints c) {
        c.gridx = 1;
        c.gridy++;
        c.weightx = RIGHT_COLUMN_WEIGHT;
        c.insets = new Insets(-10,0,10,10);
        for (FileChooserPanel fileChooser : jarLocations.subList(1, 10)) {
            content.add(fileChooser, c);
            c.gridy++;
        }
    }

    private List<FileChooserPanel> createJarLocationList() {
        List<FileChooserPanel> list = new ArrayList<>();
        for (int i = 0; i < 10; i++) {
            FileChooserPanel fileChooserPanel = FileChooserPanel.builder(langUtils, FILES_ONLY, mnemonicUtils)
                    .fileFilter(new FileNameExtensionFilter("JAR (.jar)", "jar")).build();
            if (i != 0) {
                fileChooserPanel.setVisible(false);
            }
            list.add(fileChooserPanel);
        }
        return list;
    }

    private void setDriverSettings(DatabaseDriver driver) {
        jdbcNameField.setText(driver.getJdbcName());
        moduleNameField.setText(driver.getModuleName());
        xaNameField.setText(driver.getXaClassName());
        directoryStructureField.setText(driver.getDirectoryStructure());
    }

    private void addJarLocation() {
        workingJarLocationIndex += 1;
        jarLocations.get(workingJarLocationIndex).setVisible(true);
        jarLocations.get(workingJarLocationIndex).requestFocus();
        if (!canAddMoreJars()) {
            jarAddButton.setEnabled(false);
        }
        if (canRemoveJars()) {
            jarRemoveButton.setEnabled(true);
        }

        // scroll to the current button location + new file selector height
        Rectangle bounds = jarAddButton.getBounds();
        bounds = new Rectangle(bounds.x, bounds.y + jarLocations.get(0).getHeight(), bounds.width, bounds.height);
        ((JComponent)jarAddButton.getParent()).scrollRectToVisible(bounds);
    }

    private boolean canRemoveJars() {
        return workingJarLocationIndex > 0;
    }

    private boolean canAddMoreJars() {
        return workingJarLocationIndex < MAX_JARS_COUNT;
    }

    private void removeJarLocation() {
        jarLocations.get(workingJarLocationIndex).setVisible(false);
        jarLocations.get(workingJarLocationIndex).setPath(System.getProperty("user.home"));
        workingJarLocationIndex -= 1;
        if (!canRemoveJars()) {
            jarRemoveButton.setEnabled(false);
        }
        if (canAddMoreJars()) {
            jarAddButton.setEnabled(true);
        }

        if (jarRemoveButton.isEnabled()) {
            jarRemoveButton.requestFocus();
        } else {
            jarAddButton.requestFocus();
        }
    }

    private List<String> getWorkingJarLocations() throws FileChooserPanel.InvalidPathException {
        List<String> list = new ArrayList<>();
        for (FileChooserPanel fileChooserPanel : jarLocations.subList(0, workingJarLocationIndex + 1)) {
            list.add(fileChooserPanel.asPathOrUrl());
        }
        return list;
    }

    private DatabaseDriver getSelectedDatabaseDriver() {
        return DatabaseDriver.fromName((String) driverVendors.getSelectedItem());
    }

    @Override
    public boolean prevalidate() {
        final List<String> workingJarLocations;
        try {
            workingJarLocations = getWorkingJarLocations();
        } catch (FileChooserPanel.InvalidPathException e) {
            // fall back to validation
            return true;
        }

        if (downloadUtils == null) {
            downloadUtils = new DownloadUtils(
                    new DownloadManager(),
                    (downloads, downloadManager) -> {
                        final DownloadDisplayPanel panel = new DownloadDisplayPanel(
                                downloads,
                                downloadManager, langUtils);
                        return new DownloadUtils.DownloadUI() {
                            @Override
                            public boolean display() {
                                return InstallerDialogs.showDownloadMessage(content.getRootPane(), panel, langUtils);
                            }

                            @Override
                            public DownloadHandler getDownloadHandler() {
                                return panel;
                            }
                        };
                    },
                    (download) -> InstallerDialogs.showErrorMessage(langUtils.getString(InstallerDialogs.DOWNLOAD_UNREACHABLE, download.getUrl().toString()),
                            langUtils, content));
        }

        final List<String> resolvedJarPaths = downloadUtils.downloadAll(workingJarLocations);

        if (resolvedJarPaths == null) {
            return false;
        }

        this.resolvedJarLocations = resolvedJarPaths;

        return true;
    }

    @Override
    public ValidationResult validate() {
        try {
            return new JDBCDriverValidator(langUtils).validate(getWorkingJarLocations(), resolvedJarLocations,
                    getSelectedDatabaseDriver().getXaClassName());
        } catch (FileChooserPanel.InvalidPathException e) {
            return PathValidator.newInstance("generic", langUtils).validate(e.getPath());
        }
    }

    @Override
    public void record(InstallationData installationData) {
        try {
            JDBCDriverConfig config = new JDBCDriverConfig(
                    getSelectedDatabaseDriver(),
                    getWorkingJarLocations(),
                    resolvedJarLocations
            );
            installationData.putConfig(config);
            installationData.addPostInstallTask(PostInstallTask.InstallJDBCDriver);
            installationData.addPostInstallTask(PostInstallTask.AddJDBCDriverModule);

            final HashMap<String, String> attrs = new HashMap<>();
            attrs.put(langUtils.getString(VENDOR_KEY), config.getDatabaseDriver().getVendorName());
            attrs.put(langUtils.getString(JAR_LOCATIONS_KEY), String.join(",", config.getJarList()));

            installationData.updateSummary(JDBCDriverScreen.NAME, langUtils.getString(ConfigureRuntimeEnvironmentScreen.JDBC_DRIVER_KEY), attrs);
        } catch (FileChooserPanel.InvalidPathException e) {
            // handled in validate
            throw new RuntimeException(e);
        }
    }

    @Override
    public void rollback(InstallationData installationData) {
        installationData.removeConfig(JDBCDriverConfig.class);
        installationData.removePostInstallTask(PostInstallTask.InstallJDBCDriver);
        installationData.removePostInstallTask(PostInstallTask.AddJDBCDriverModule);
    }
}
