/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2024 Red Hat, Inc., and individual contributors
 * as indicated by the @author tags.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jboss.installer.postinstall.task;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.BiFunction;
import java.util.stream.Collectors;

import org.jboss.installer.auto.AutomaticInstallationParsingException;
import org.jboss.installer.core.DatabaseDriver;
import org.jboss.installer.core.FlatListPostInstallConfig;

public class JDBCDriverConfig extends FlatListPostInstallConfig {

    private DatabaseDriver databaseDriver;
    private List<String> jarList;
    // not serialized
    private List<String> resolvedJarList;

    public JDBCDriverConfig(){

    }

    public JDBCDriverConfig(DatabaseDriver databaseDriver, List<String> jarList, List<String> resolvedJarLocations) {
        this.databaseDriver = databaseDriver;
        this.jarList = jarList;
        this.resolvedJarList = resolvedJarLocations;
    }

    public DatabaseDriver getDatabaseDriver() {
        return databaseDriver;
    }

    public List<String> getJarList() {
        return jarList;
    }

    public List<String> getResolvedJarList() {
        return resolvedJarList;
    }

    @Override
    protected Map<String, String> listAttributes() {
        HashMap<String, String> attributes = new HashMap<>();
        attributes.put("databaseDriver", databaseDriver.name());
        for (int i = 0; i < jarList.size(); i++) {
            attributes.put("jarPath-" + i, jarList.get(i));
        }
        return attributes;
    }

    @Override
    protected void acceptAttributes(Map<String, String> attributes, BiFunction<String, String, String> variableResolver) throws AutomaticInstallationParsingException {
        databaseDriver = DatabaseDriver.valueOf(attributes.get("databaseDriver"));
        jarList = getListOfJarPaths(attributes);

    }

    @Override
    public int hashCode() {
        return Objects.hash(databaseDriver, jarList);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) return true;
        if (obj == null || getClass() != obj.getClass()) return false;
        JDBCDriverConfig config = (JDBCDriverConfig) obj;
        return Objects.equals(databaseDriver, config.databaseDriver) && Objects.equals(jarList, config.jarList);
    }

    private List<String> getListOfJarPaths(Map<String, String> attributes) {
        return attributes.entrySet().stream()
                .filter(entry -> entry.getKey().startsWith("jarPath"))
                .map(Map.Entry::getValue)
                .collect(Collectors.toList());
    }

    public boolean hasResolvedJarList() {
        return resolvedJarList != null && !resolvedJarList.isEmpty();
    }

    public void setResolvedJarList(List<String> resolvedJarList) {
        this.resolvedJarList = resolvedJarList;
    }
}
