/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * <p>
 * http://www.apache.org/licenses/LICENSE-2.0
 * <p>
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jboss.installer.screens.domain;

import org.jboss.installer.common.UiResources;
import org.jboss.installer.core.LanguageUtils;
import org.jboss.installer.postinstall.task.secdom.JdbcConfig;

import javax.swing.JLabel;
import javax.swing.JComboBox;
import javax.swing.JPanel;
import javax.swing.JTextField;

import java.awt.Insets;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;

import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.stream.Collectors;

public class DatabaseSecurityDomainPanel extends JPanel {

    public static final String SQL_QUERY_LABEL = "security_domain_screen.jdbc.query.label";
    public static final String JDBC_DATASOURCE_LABEL = "security_domain_screen.jdbc.datasource.label";
    public static final String JDBC_MAPPER_LABEL = "security_domain_screen.jdbc.mapper.label";
    private final LanguageUtils langUtils;
    private final JComboBox datasourceField;
    private final JTextField sqlQuery;
    private final JComboBox mapperTypeField;
    private final SelectorPanel selectorPanel;

    DatabaseSecurityDomainPanel(LanguageUtils languageUtils) {
        this.langUtils = languageUtils;

        this.setLayout(new GridBagLayout());

        datasourceField = UiResources.createDropdown(new String[]{"ExampleDS"});
        sqlQuery = UiResources.createTextField("");
        mapperTypeField = UiResources.createDropdown(Arrays.stream(JdbcConfig.MapperType.values())
                .map(JdbcConfig.MapperType::getText).collect(Collectors.toList())
                .toArray(new String[]{}));
        selectorPanel = new SelectorPanel(langUtils);

        initUi();
    }

    private void initUi() {
        final GridBagConstraints c = UiResources.initializeConstraints();
        c.insets = new Insets(5, 0, 0, 0);

        c.gridwidth = 1;
        final JLabel datasourceLabel = UiResources.createFieldLabelWithColonSuffix(langUtils.getString(JDBC_DATASOURCE_LABEL));
        this.add(datasourceLabel, c);
        c.gridx = 1;
        this.add(datasourceField, c);
        UiResources.setTooltip(datasourceLabel, langUtils.getString(JDBC_DATASOURCE_LABEL + ".tooltip"));
        UiResources.setTooltip(datasourceField, langUtils.getString(JDBC_DATASOURCE_LABEL + ".tooltip"));
        c.gridx = 0;
        c.gridwidth = 2;

        c.gridy++;
        final JLabel sqlQueryLabel = UiResources.createFieldLabelWithColonSuffix(langUtils.getString(SQL_QUERY_LABEL));
        this.add(sqlQueryLabel, c);
        c.gridy++;
        this.add(sqlQuery, c);
        UiResources.setTooltip(sqlQueryLabel, langUtils.getString(SQL_QUERY_LABEL + ".tooltip"));
        UiResources.setTooltip(sqlQuery, langUtils.getString(SQL_QUERY_LABEL + ".tooltip"));

        c.gridwidth = 1;
        c.gridy++;
        final JLabel mapperTypeLabel = UiResources.createFieldLabelWithColonSuffix(langUtils.getString(JDBC_MAPPER_LABEL));
        this.add(mapperTypeLabel, c);
        c.gridx = 1;
        this.add(mapperTypeField, c);
        UiResources.setTooltip(mapperTypeLabel, langUtils.getString(JDBC_MAPPER_LABEL + ".tooltip"));
        UiResources.setTooltip(mapperTypeField, langUtils.getString(JDBC_MAPPER_LABEL + ".tooltip"));
        c.gridx = 0;
        c.gridwidth = 2;

        c.gridy++;
        mapperTypeField.addActionListener(e -> selectorPanel.mapperChanged(JdbcConfig.MapperType.from((String) mapperTypeField.getSelectedItem())));
        this.add(selectorPanel, c);
    }

    JdbcConfig getConfig() {
        final JdbcConfig jdbcConfig = new JdbcConfig();
        jdbcConfig.setDatasourceName((String) datasourceField.getSelectedItem());
        jdbcConfig.setSqlQuery(UiResources.readTextValue(sqlQuery));
        jdbcConfig.setMapperType(JdbcConfig.MapperType.from((String) mapperTypeField.getSelectedItem()));
        selectorPanel.recordUpdates(jdbcConfig);
        return jdbcConfig;
    }

    Map<String, String> getAttributes() {
        final HashMap<String, String> attrs = new HashMap<>();
        final JdbcConfig config = getConfig();

        attrs.put(langUtils.getString(SelectorPanel.GROUP_SELECTOR_LABEL), config.getGroupIndex());
        attrs.put(langUtils.getString(SelectorPanel.PASSWORD_SELECTOR_LABEL), config.getPasswordIndex());
        attrs.put(langUtils.getString(SQL_QUERY_LABEL), config.getSqlQuery());
        attrs.put(langUtils.getString(JDBC_DATASOURCE_LABEL), config.getDatasourceName());
        attrs.put(langUtils.getString(JDBC_MAPPER_LABEL), config.getMapperType().getText());

        if (config.getMapperType() == JdbcConfig.MapperType.Digest) {
            attrs.put(langUtils.getString(SelectorPanel.HASH_ENC_LABEL), config.getHashEncoding());
            attrs.put(langUtils.getString(SelectorPanel.HASH_ALG_LABEL), config.getEncodingAlgorithm());
        }
        if (config.getMapperType() == JdbcConfig.MapperType.SaltedDigest) {
            attrs.put(langUtils.getString(SelectorPanel.HASH_ENC_LABEL), config.getHashEncoding());
            attrs.put(langUtils.getString(SelectorPanel.HASH_ALG_LABEL), config.getEncodingAlgorithm());
            attrs.put(langUtils.getString(SelectorPanel.SALT_SELECTOR_LABEL), config.getSaltIndex());
            attrs.put(langUtils.getString(SelectorPanel.SALT_ENCODING_LABEL), config.getSaltEncoding());
        }
        return attrs;
    }
}
