/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2024 Red Hat, Inc., and individual contributors
 * as indicated by the @author tags.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jboss.installer.postinstall.task;

import static org.junit.Assert.assertEquals;

import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.Optional;

import org.jboss.installer.auto.InstallationDataSerializer;
import org.jboss.installer.core.InstallationData;
import org.jboss.installer.test.utils.MockLanguageUtils;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

public class PortConfigurationDescriptorTest {

    @Rule
    public TemporaryFolder temp = new TemporaryFolder();

    @Test
    public void testSerializeDeserialize() throws Exception {
        final InstallationDataSerializer serializer = new InstallationDataSerializer(new MockLanguageUtils());
        // set up test idata
        final InstallationData idata = new InstallationData();
        final PortBindingConfig config = new PortBindingConfig(100);
        final PortBindingConfig.SocketBinding mgtIntBinding = new PortBindingConfig.SocketBinding(null, "jboss.management.http.port", 10090);
        config.setManagementInterfacePort(mgtIntBinding);
        final PortBindingConfig.SocketBinding httpDomainBinding = new PortBindingConfig.SocketBinding("http", "jboss.http.port", 8081);
        final PortBindingConfig.SocketBinding httpsDomainBinding = new PortBindingConfig.SocketBinding("https", "jboss.https.port", 8481);
        final PortBindingConfig.SocketBinding httpStandaloneBinding = new PortBindingConfig.SocketBinding("http", "jboss.http.port", 8082);
        config.setPorts("domain-ha", Arrays.asList(httpDomainBinding, httpsDomainBinding));
        config.setPorts("standalone", Arrays.asList(httpStandaloneBinding));
        idata.putConfig(config);

        // serialize and deserialize the idata
        final Path tempFile = temp.newFile("auto.xml").toPath();
        serializer.serialize(idata, tempFile);
        final InstallationData loadedData = serializer.deserialize(tempFile, Optional.empty());

        // validate the same configs
        final PortBindingConfig loadedConfig = loadedData.getConfig(PortBindingConfig.class);
        assertEquals(100, loadedConfig.getOffset());
        assertEquals(mgtIntBinding, loadedConfig.getManagementInterface());
        assertEquals(httpDomainBinding, loadedConfig.getPorts("domain-ha").get(0));
        assertEquals(httpsDomainBinding, loadedConfig.getPorts("domain-ha").get(1));
        assertEquals(httpStandaloneBinding, loadedConfig.getPorts("standalone").get(0));
    }

    @Test
    public void testSerializeDeserializeWithoutManagementInterface() throws Exception {
        final InstallationDataSerializer serializer = new InstallationDataSerializer(new MockLanguageUtils());
        // set up test idata
        final InstallationData idata = new InstallationData();
        final PortBindingConfig config = new PortBindingConfig(100);
        final PortBindingConfig.SocketBinding httpDomainBinding = new PortBindingConfig.SocketBinding("http", "jboss.http.port", 8081);
        final PortBindingConfig.SocketBinding httpsDomainBinding = new PortBindingConfig.SocketBinding("https", "jboss.https.port", 8481);
        final PortBindingConfig.SocketBinding httpStandaloneBinding = new PortBindingConfig.SocketBinding("http", "jboss.http.port", 8082);
        config.setPorts("domain-ha", Arrays.asList(httpDomainBinding, httpsDomainBinding));
        config.setPorts("standalone", Arrays.asList(httpStandaloneBinding));
        idata.putConfig(config);

        // serialize and deserialize the idata
        final Path tempFile = temp.newFile("auto.xml").toPath();
        serializer.serialize(idata, tempFile);
        System.out.println(Files.readString(tempFile));
        final InstallationData loadedData = serializer.deserialize(tempFile, Optional.empty());

        // validate the same configs
        final PortBindingConfig loadedConfig = loadedData.getConfig(PortBindingConfig.class);
        assertEquals(100, loadedConfig.getOffset());
        assertEquals(httpDomainBinding, loadedConfig.getPorts("domain-ha").get(0));
        assertEquals(httpsDomainBinding, loadedConfig.getPorts("domain-ha").get(1));
        assertEquals(httpStandaloneBinding, loadedConfig.getPorts("standalone").get(0));
    }
}
