/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2021 Red Hat, Inc., and individual contributors
 * as indicated by the @author tags.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jboss.installer.postinstall.server;

import org.jboss.dmr.ModelNode;

import java.nio.file.Path;
import java.util.Optional;
import java.util.Set;

public interface EmbeddedServer extends AutoCloseable {
    /**
     * Start embedded server using {@code config}. The @{code config} has to be available in the server's configuration directory.
     *
     * @param config - configuration file to use
     * @throws ServerOperationException - if the server failed to start
     */
    void start(String config) throws ServerOperationException;

    /**
     * Stop embedded server.
     *
     * @throws ServerOperationException - if unable to execute the shutdown command
     */
    void shutdown() throws ServerOperationException;

    /**
     * Executes operation on the server. The server has to be started in order to accept operations.
     *
     * @param op - operation to run
     * @param desc - description of the operation (for debug)
     * @return - {@code result} part of the response
     * @throws ServerOperationException - if the operation returned {@outcome failure}. The failure details are available in the exception
     */
    ModelNode execute(ModelNode op, String desc) throws ServerOperationException;

    /**
     * Lists all available configurations in the server's configuration directory
     * @return
     */
    Set<String> availableConfigurations();

    String currentConfiguration();

    boolean isDomain();

    ServerPath toRelativePathNode(Path file);

    boolean aliasExists(String alias, String storeName) throws ServerOperationException;

    ModelNode encryptPassword(String alias, String storeName, String keystorePassword) throws ServerOperationException;

    ModelNode relativise(Path path, ModelNode node);

    Optional<Path> getJBossCliXml();

    class ServerPath {
        private final Optional<String> relativeTo;
        private final String path;

        public ServerPath(String relativeTo, String path) {
            this.relativeTo = Optional.of(relativeTo);
            this.path = path;
        }

        public ServerPath(String path) {
            this.relativeTo = Optional.empty();
            this.path = path;
        }

        public Optional<String> getRelativeTo() {
            return relativeTo;
        }

        public String getPath() {
            return path;
        }
    }


}
