/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.web.tomcat.service.session.distributedcache.spi;

import java.util.Map;
import java.util.Set;

/**
 * SPI implemented by the distributed caching layer.
 * 
 * @author Brian Stansberry
 * @version $Revision: $
 */
public interface DistributedCacheManager<T extends OutgoingDistributableSessionData>
{
   /**
    * Starts the distributed caching layer.
    */
   void start();

   /**
    * Stops the distributed caching layer.
    */
   void stop();

   /**
    * Gets the BatchingManager.
    */
   BatchingManager getBatchingManager();

   /**
    * Notification to the distributed cache that a session has been
    * newly created.
    * 
    * @param realId the session id with any appended jvmRoute info removed
    */
   void sessionCreated(String realId);
   
   /**
    * Store or update a session in the distributed cache.
    * 
    * @param session the session
    */
   void storeSessionData(T sessionData);

   /**
    * Globally remove a session from the distributed cache.
    * 
    * @param realId the session's id, excluding any jvmRoute
    */
   void removeSession(String realId);

   /**
    * Remove a session from the distributed cache on this node only.
    * 
    * @param realId the session's id, excluding any jvmRoute
    */
   void removeSessionLocal(String realId);

   /**
    * Remove a non-locally active session from the distributed cache, but on 
    * this node only.
    * 
    * @param realId the session's id, excluding any jvmRoute
    * @param dataOwner identifier of node where the session is active
    */
   void removeSessionLocal(String realId, String dataOwner);

   /**
    * Evict a session from the in-memory portion of the distributed cache, on
    * this node only.
    * 
    * @param realId the session's id, excluding any jvmRoute
    */
   void evictSession(String realId);

   /**
    * Evict a non-locally-active session from the in-memory portion of the 
    * distributed cache, on this node only.
    * 
    * @param realId the session's id, excluding any jvmRoute
    * @param dataOwner identifier of node where the session is active
    */
   void evictSession(String realId, String dataOwner);

   /**
    * Get the {@link IncomingDistributableSessionData} that encapsulates the distributed
    * cache's information about the given session.
    * 
    * @param realId the session's id, excluding any jvmRoute
    * @param initialLoad <code>true</code> if this is the first access of
    *                    this session's data on this node
    *                          
    * @return the session data
    */
   IncomingDistributableSessionData getSessionData(String realId, boolean initialLoad);

   /**
    * Get the {@link IncomingDistributableSessionData} that encapsulates the distributed
    * cache's information about the given session.
    * 
    * @param realId the session's id, excluding any jvmRoute
    * @param dataOwner identifier of node where the session is active; 
    *                  <code>null</code> if locally active or location where
    *                  active is unknown
    * @param includeAttributes should 
    *                          @link IncomingDistributableSessionData#providesSessionAttributes()}
    *                          return <code>true</code>?
    *                          
    * @return the session data
    */
   IncomingDistributableSessionData getSessionData(String realId, String dataOwner, boolean includeAttributes);

   /**
    * Gets the ids of all sessions in the underlying cache.
    *
    * @return Map<String, String> containing all of the session ids of sessions in the cache
    *         (with any jvmRoute removed) as keys, and the identifier of the data owner for
    *         the session as value (or a <code>null</code>  value if buddy
    *         replication is not enabled.) Will not return <code>null</code>.
    */
   Map<String, String> getSessionIds();

   /**
    * Gets whether the underlying cache supports passivation.
    */
   boolean isPassivationEnabled();
   
   /**
    * Gets whether this DistributedCacheManager supports operations on
    * session attributes; if <code>false</code> the various attribute
    * operations in will throw {@link UnsupportedOperationException}.
    */
   boolean getSupportsAttributeOperations();

   /**
    * Get the value of the attribute with the given key from the given session.
    * 
    * @param realId the session id with any jvmRoute removed
    * @param key the attribute key
    * @return the attribute value, or <code>null</code>
    * 
    * @throws UnsupportedOperationException if {@link #getSupportsAttributeOperations()}
    *                                       would return <code>false</code>
    */
   Object getAttribute(String realId, String key);

   /**
    * Stores the given value under the given key in the given session.
    * 
    * @param realId the session id with any jvmRoute removed
    * @param key the attribute key
    * @param value the previous attribute value, or <code>null</code>
    * 
    * @throws UnsupportedOperationException if {@link #getSupportsAttributeOperations()}
    *                                       would return <code>false</code>
    */
   void putAttribute(String realId, String key, Object value);

   /**
    * Stores the given map of attribute key/value pairs in the given session.
    * 
    * @param realId the session id with any jvmRoute removed
    * @param map
    * 
    * @throws UnsupportedOperationException if {@link #getSupportsAttributeOperations()}
    *                                       would return <code>false</code>
    */
   void putAttribute(String realId, Map<String, Object> map);

   /**
    * Removes the attribute with the given key from the given session.
    * 
    * @param realId the session id with any jvmRoute removed
    * @param key the attribute key
    * @return the previous attribute value, or <code>null</code>
    * 
    * @throws UnsupportedOperationException if {@link #getSupportsAttributeOperations()}
    *                                       would return <code>false</code>
    */
   Object removeAttribute(String realId, String key);

   /**
    * Removes the attribute with the given key from the given session, but 
    * only on the local node.
    * 
    * @param realId the session id with any jvmRoute removed
    * @param key the attribute key
    * 
    * @throws UnsupportedOperationException if {@link #getSupportsAttributeOperations()}
    *                                       would return <code>false</code>
    */
   void removeAttributeLocal(String realId, String key);

   /**
    * Obtain the attribute keys associated with this session.
    * 
    * @param realId the session id with any jvmRoute removed
    * @return the attribute keys or an empty Set if none are found
    * 
    * @throws UnsupportedOperationException if {@link #getSupportsAttributeOperations()}
    *                                       would return <code>false</code>
    */
   Set<String> getAttributeKeys(String realId);

   /**
    * Return all attributes associated with this session id.
    * 
    * @param realId the session id with any jvmRoute removed
    * @return the attributes, or any empty Map if none are found.
    * 
    * @throws UnsupportedOperationException if {@link #getSupportsAttributeOperations()}
    *                                       would return <code>false</code>
    */
   Map<String, Object> getAttributes(String realId);

}