package org.hibernate.plugins;

import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.List;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.Properties;
import java.net.URL;
import java.net.URLClassLoader;
import javax.xml.parsers.SAXParserFactory;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.ParserConfigurationException;

import org.apache.maven.project.MavenProject;
import org.hibernate.tool.hbm2ddl.Target;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;
import org.xml.sax.helpers.DefaultHandler;
import org.hibernate.cfg.Configuration;
import org.hibernate.ejb.Ejb3Configuration;
import org.hibernate.tool.hbm2ddl.SchemaExport;

public abstract class AbstractJPA2SchemaExport extends AbstractMojo
{
    public final static String DIALECT = "hibernate.dialect";

    /**
     * The maven project.
     *
     * @parameter expression="${project}"
     * @required
     * @readonly
     */
    private MavenProject project;

    /**
     * The hibernate SQL dialect.
     *
     * @parameter expression="${hibernate.dialect}" default-value="org.hibernate.dialect.H2Dialect"
     */
    private String dialect;

    /**
     * Export target
     *
     * @parameter expression="${hibernate.target}" default-value="EXPORT"
     */
    private String target;

     /**
     * Export type
     *
     * @parameter expression="${hibernate.type}" default-value="BOTH"
     */
    private String type;

    /**
     * Format the generated SQL nicely in the script
     *
     * @parameter expression="${hibernate.formatsql}" default-value="false"
     */
    private boolean formatsql;

    /**
     * Set an end of line delimiter for the script
     *
     * @parameter expression="${hibernate.delimiter}"
     */
    private String delimiter;

    /**
     * Set haltOnError processing for SchemaExport
     *
     * @parameter expression="${hibernate.haltOnError}" default-value="true"
     */
    private boolean haltOnError;

    /**
     * Set haltOnNoDescriptor
     *
     * @parameter expression="${hibernate.haltOnNoDescriptor}" default-value="true"
     */
    private boolean haltOnNoDescriptor;

    /**
     * @parameter expression="${hibernate.output.directory}" default-value="${project.build.directory}"
     */
    private String outputDirectory;

    protected abstract List<String> getClasspathElements();

    protected List<String> getPersistenceUnitNames(InputStream persistenceDescriptor)
    throws MojoExecutionException
    {
        List<String> persistentUnitNames = null;
        SAXParserFactory factory = SAXParserFactory.newInstance();
        factory.setValidating(true);
        SAXParser parser = null;

        try
        {
            parser = factory.newSAXParser();
        }
        catch (ParserConfigurationException e)
        {
            getLog().error("\nUnable to make a parse with given configuration!", e);
            throw new MojoExecutionException(e.getMessage());
        }
        catch (SAXException e)
        {
            getLog().error("\nUnable to create parser factory!", e);
            throw new MojoExecutionException(e.getMessage());
        }

        PersistentUnitHandler persistentUnitHandler = new PersistentUnitHandler();
        try
        {
            parser.parse(persistenceDescriptor, persistentUnitHandler);
            persistentUnitNames = persistentUnitHandler.getPersistenceUnitNames();
        }
        catch (IOException e)
        {
            getLog().error("\nCould not initialize parser!");
            throw new MojoExecutionException(e.getMessage());
        }
        catch (SAXParseException e)
        {
            getLog().error("\nParsing error!\n" + "  line = " + e.getLineNumber() + "\n" + "  location = " + e.getSystemId());
            throw new MojoExecutionException(e.getMessage());
        }
        catch (SAXException e)
        {
            getLog().error("\nCould not initialize parser!");
            throw new MojoExecutionException(e.getMessage());
        }

        return persistentUnitNames;
    }

    private SchemaExport.Type interpretType(String type)
    throws MojoExecutionException
    {
        SchemaExport.Type exportType;
        try
        {
            exportType = SchemaExport.Type.valueOf(type);
        }
        catch (IllegalArgumentException e)
        {
            throw new MojoExecutionException("\nNot a valid export type!");
        }
        catch (NullPointerException e)
        {
            throw new MojoExecutionException("\nNo export type set!");
        }
        return exportType;
    }

    private Target interpretTarget(String target)
    throws MojoExecutionException
    {
        Target exportTarget;
        try
        {
            exportTarget = Target.valueOf(target);
        }
        catch (IllegalArgumentException e)
        {
            throw new MojoExecutionException("\nNot a valid export target!");
        }
        catch (NullPointerException e)
        {
            throw new MojoExecutionException("\nNo export target set!");
        }
        return exportTarget;
    }

    protected void export(ClassLoader classLoader, String persistenceUnitName,
                          String destination, boolean create, boolean format)
    throws MojoExecutionException
    {
        getLog().info("Starting schema export [" + persistenceUnitName + "]");
        Thread.currentThread().setContextClassLoader(classLoader);
        Properties properties = new Properties();
        properties.setProperty(DIALECT, dialect);
        Ejb3Configuration cfg = new Ejb3Configuration().configure(persistenceUnitName, properties);
        Configuration hbmcfg = cfg.getHibernateConfiguration();
        SchemaExport schemaExport = new SchemaExport(hbmcfg);
        schemaExport.setOutputFile(destination);
        schemaExport.setDelimiter(delimiter);
        schemaExport.setFormat(format);
        schemaExport.setHaltOnError(haltOnError);
        schemaExport.execute(interpretTarget(target), interpretType(type));
        getLog().info("Schema exported to " + destination);
    }

    protected String createFileName(String persistenceUnitName)
    {
        // The GAV
        String groupId = project.getGroupId();
        String artifactId = project.getArtifactId();
        String version = project.getVersion();
        //String gav = groupId + "-" + artifactId + "-" + version;
        String prefix = artifactId;

        String fileName = prefix + "." +  persistenceUnitName + "." +
                dialect.substring(dialect.lastIndexOf('.') + 1) + ".sql";

        return fileName;
    }

    public void execute() throws MojoExecutionException, MojoFailureException
    {
        ClassLoader classLoader = null;
        try
        {
            getLog().debug("Creating ClassLoader for project-dependencies...");
            List<String> classpathFiles = getClasspathElements();
            URL[] urls = new URL[classpathFiles.size()];
            for (int i = 0; i < classpathFiles.size(); ++i)
            {
                getLog().debug("Dependency: " + classpathFiles.get(i));
                urls[i] = new File(classpathFiles.get(i)).toURI().toURL();
            }
            classLoader = new URLClassLoader(urls, getClass().getClassLoader());
        }
        catch (Exception e)
        {
            getLog().error("Error while creating ClassLoader!", e);
            throw new MojoExecutionException(e.getMessage());
        }

        InputStream is = classLoader.getResourceAsStream("META-INF/persistence.xml");
        if (is != null)
        {
            List<String> persistenceUnitNames = getPersistenceUnitNames(is);
            Iterator<String> pnIter = persistenceUnitNames.iterator();
            File outputDir = new File(outputDirectory);
            if (!outputDir.exists()) outputDir.mkdirs();
            while (pnIter.hasNext())
            {
                String persistenceUnitName = pnIter.next();
                String destination = outputDirectory + File.separator + createFileName(persistenceUnitName);
                export(classLoader, persistenceUnitName, destination, false, true);
            }
        }
        else
        {
            if (haltOnNoDescriptor)
            {
                throw new MojoExecutionException("\nNo JPA persistence.xml found");
            }
            else
            {
                getLog().info("\nNo JPA persistence.xml, skipping export");
            }
        }
    }

    protected class PersistentUnitHandler extends DefaultHandler
    {
        private final String PERSISTENT_UNIT = "persistence-unit";
        private final String PERSISTENT_UNIT_NAME = "name";
        private List<String> persistentUnitNames = new ArrayList<String>();

        @Override
        public void startElement(String namespaceURI, String localName, String qName, Attributes atts) throws SAXException
        {
            if (qName.equals(PERSISTENT_UNIT))
            {
                persistentUnitNames.add(atts.getValue(PERSISTENT_UNIT_NAME));
            }
        }

        List<String> getPersistenceUnitNames()
        {
            return persistentUnitNames;
        }
    }
}
