package org.hibernate.plugins.guards;

import org.hibernate.plugins.util.Util;

import java.util.HashMap;
import java.util.Map;

/**
 * @author vdedik@redhat.com
 */
public abstract class AbstractGuard implements Guard {

    public static final Map<String, String> TOKENS = new HashMap<String, String>() {{
        put("create table", "(?i)create\\s+table\\s+(\\w+)");
        put("drop table", "(?i)drop\\s+table\\s+(\\w+)");
        put("create sequence", "(?i)create\\s+sequence\\s+(\\w+)");
        put("drop sequence", "(?i)drop\\s+sequence\\s+(\\w+)");
    }};

    @Override
    public String safeGuard(String sql) {
        String result = sql;
        if (Util.statementContains(sql, "create table")) {
            result = safeGuardCreateTable(sql);
        } else if (Util.statementContains(sql, "create sequence")) {
            result = safeGuardCreateSequence(sql);
        } else if (Util.statementContains(sql, "drop table")) {
            result = safeGuardDropTable(sql);
        } else if (Util.statementContains(sql, "drop sequence")) {
            result = safeGuardDropSequence(sql);
        } else if (Util.statementContains(sql, "alter table")) {
            result = safeGuardAlterTable(sql);
        }

        return result;
    }

    /**
     * Safe guard for create table statement
     */
    public abstract String safeGuardCreateTable(String sql);

    /**
     * Safe guard for create sequence statement
     */
    public String safeGuardCreateSequence(String sql) {
        throw new IllegalArgumentException("This dialect of database does not support sequences");
    }

    /**
     * Safe guard for drop table statement
     */
    public abstract String safeGuardDropTable(String sql);

    /**
     * Safe guard for drop sequence statement
     */
    public String safeGuardDropSequence(String sql) {
        throw new IllegalArgumentException("This dialect of database does not support sequences");
    }

    /**
     * Safe guard for alter table statement
     */
    public abstract String safeGuardAlterTable(String sql);

    /**
     * Decorator for executing procedure
     */
    public  String decorateWithCall(String sql) {
        return "call " + sql;
    }

    /**
     * Decorator for procedure execute_if_exists
     */
    public String decorateWithExecuteIf(String sql) {
        if (Util.statementContains(sql, "execute_if_exists")) {
            return sql;
        }
        return decorateWithCall("execute_if_exists('" + Util.removeSemicolon(sql) + "')") + Util.SQL_DELIMITER;
    }

    /**
     * Decorator for procedure execute_if_not_exists
     */
    public String decorateWithExecuteIfNot(String sql) {
        if (Util.statementContains(sql, "execute_if_not_exists")) {
            return sql;
        }
        return decorateWithCall("execute_if_not_exists('" + Util.removeSemicolon(sql) + "')") + Util.SQL_DELIMITER;
    }

    /**
     * Decorator for "if not exists" guard
     */
    public String decorateWithIfNot(String token, String sql) {
        if (Util.statementContains(sql, "if not exists") || !TOKENS.containsKey(token)) {
            return sql;
        }
        return sql.replaceAll(TOKENS.get(token), token + " if not exists $1");
    }

    /**
     * Decorator for "if exists" guard
     */
    public String decorateWithIf(String token, String sql) {
        if (Util.statementContains(sql, "if exists") ||  !TOKENS.containsKey(token)) {
            return sql;
        }
        return sql.replaceAll(TOKENS.get(token), token + " if exists $1");
    }
}
