package org.jboss.ip.dbtool;

/**
 * Provides static methods for creating a database connection URL.
 *
 * @author Alex Creasy <acreasy@redhat.com>
 */
public final class ConnectionUrlCreator {

    public static final String HOSTNAME_VAR = "\\$\\{HOSTNAME\\}";
    public static final String PORT_VAR = "\\$\\{PORT\\}";
    public static final String DBNAME_VAR = "\\$\\{DATABASE_NAME\\}";
    public static final String RAC_VAR = "\\$\\{HOSTSPORTS\\}";

    // ORACLE RAC SPECIFIC
    public static final String ORACLE_RAC_IDENTIFIER = "oraclerac";

    public static final String ORACLE_RAC_URL_PATTERN = "jdbc:oracle:thin:@(DESCRIPTION=(LOAD_BALANCE=on)" +
            "${HOSTSPORTS}(CONNECT_DATA=(SERVICE_NAME=${DATABASE_NAME})))";
    public static final String ORACLE_RAC_HOSTSPORTS_PATTERN = "(ADDRESS=(PROTOCOL=TCP)(HOST=${HOSTNAME})" +
            "(PORT=${PORT}))";


    /**
     * Creates a connection URL to non Oracle RAC database servers.
     *
     * @param db the Database instance that will be connected to (provides the correct format for the URL)
     * @param dbName the database schema (table space) name.
     * @param hostname the hostname of the database server
     * @param port the port of the datbase server.
     * @return a correctly formatted connection URL
     * @throws IllegalArgumentException if Any String parameters are empty Strings.
     */
    public static String createUrl(final Database db, final String dbName, final String hostname, final String port) {

        if (dbName.isEmpty())
            throw new IllegalArgumentException("Empty param: dbName");

        if (hostname.isEmpty())
            throw new IllegalArgumentException("Empty param: hostname");

        if (port.isEmpty())
            throw new IllegalArgumentException("Empty param: port");

        String connUrl;
        connUrl = db.getConnectionUrlPattern();
        connUrl = connUrl.replaceFirst(HOSTNAME_VAR, hostname);
        connUrl = connUrl.replaceFirst(PORT_VAR, port);
        return connUrl.replaceFirst(DBNAME_VAR, dbName);
    }

    /**
     * Creates a connection URL for an Oracle RAC database cluster.
     *
     * @param dbName the database schema (table space) name
     * @param hostnames the hostnames of each database in the cluster in [hostname]:[port] format
     * @return a correctly formatted Oracle RAC connection url.
     * @throws IllegalArgumentException if an empty String is given as dbName or less than 1 hostname is provided
     */
    public static String createOracleRacUrl(final String dbName, final String... hostnames) {

        if (dbName.isEmpty())
            throw new IllegalArgumentException("Empty argument: dbName");

        if (hostnames.length < 1)
            throw new IllegalArgumentException("Method requires a minimum of 1 hostnames");

        final StringBuilder sb = new StringBuilder();
        for (String hostname : hostnames) {

            // extract hostname and port.
            final String[] x = hostname.split(":");

            sb.append(ORACLE_RAC_HOSTSPORTS_PATTERN.replaceFirst(HOSTNAME_VAR, x[0])
                    .replaceFirst(PORT_VAR, x[1]));
        }
        return ORACLE_RAC_URL_PATTERN.replaceFirst(RAC_VAR, sb.toString()).replaceFirst(DBNAME_VAR, dbName);
    }

    // Suppress default constructor to prevent instantiation
    private ConnectionUrlCreator() {}
}
