package com.izforge.izpack.panels;

import java.awt.BorderLayout;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.LayoutManager2;
import java.io.File;
import java.util.Arrays;

import javax.accessibility.AccessibleContext;
import javax.swing.AbstractButton;
import javax.swing.ButtonGroup;
import javax.swing.ButtonModel;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JButton;
import javax.swing.JRadioButton;
import javax.swing.JTextArea;
import javax.swing.SwingConstants;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import com.izforge.izpack.adaptator.IXMLElement;
import com.izforge.izpack.gui.IzPanelLayout;
import com.izforge.izpack.gui.LabelFactory;
import com.izforge.izpack.installer.InstallData;
import com.izforge.izpack.installer.InstallerFrame;
import com.izforge.izpack.installer.IzPanel;
import com.izforge.izpack.util.AbstractUIHandler;
import com.izforge.izpack.util.GUIHelper;
import com.izforge.izpack.util.IoHelper;

public class MavenRepoCheckPanel extends IzPanel 
{

    private static final long serialVersionUID = 1L;
    protected RepoCheckPanel repoCheckPanel;
    private static final int MAX_TEXT_JPN  = 80;
    private static final int MAX_TEXT_CHN  = 100;
    private static final int MAX_TEXT_ALL  = 115;
    public MavenRepoCheckPanel(InstallerFrame parent, InstallData idata)
    {
        this(parent, idata, new IzPanelLayout());
    }

    public MavenRepoCheckPanel(InstallerFrame parent, InstallData idata, LayoutManager2 layout)
    {
        super(parent, idata, layout);
        setBorder(GUIHelper.createIzPackBorder());
        setLayout(new BorderLayout());       
        
        JPanel mainPanel = GUIHelper.createMainPanel(!parent.hasBackground);
        
        JLabel title = LabelFactory.createTitleLabel(idata.langpack.getString("MavenRepoCheckPanel.headline"), !parent.hasBackground);
        mainPanel.add(title, GUIHelper.getTitleConstraints());
        
        repoCheckPanel = new RepoCheckPanel(this, idata, !parent.hasBackground);
        mainPanel.add(repoCheckPanel, GUIHelper.getContentPanelConstraints());
        
        // not sure why, but this scrollpane really makes the UI go crazy
        //JScrollPane scroller = GUIHelper.createPanelScroller(getBorder(), mainPanel, !parent.hasBackground);
        add(mainPanel, BorderLayout.CENTER);
    }
    
    /** Generate string for the summary panel */
    public String getSummaryBody()
    {
        String installQuickstarts = idata.getVariable("installQuickStarts");
        String installRepo = idata.getVariable("mavenSettings");
        
        if (installQuickstarts == null || installRepo == null){ return null; }
        if (installQuickstarts.equals("false")||installRepo.equals("off")) { return null; }
        
        return idata.langpack.getString("MavenRepoCheckPanel.location")
                + idata.getVariable("MAVEN_REPO_PATH") + "<br>"
                + idata.langpack.getString("MavenRepoCheckPanel.settings.location")
                + idata.getVariable("MAVEN_SETTINGS_FULLPATH");
    }

    public void makeXMLData(IXMLElement panelRoot)
    {
        new MavenRepoCheckPanelAutomationHelper().makeXMLData(idata, panelRoot);
    }

    public boolean isValidated(){
        return repoCheckPanel.validated();
    }

    
    class RepoCheckPanel extends JPanel implements ChangeListener
    {        

        private static final long serialVersionUID = -7481507528139017008L;
        protected JLabel repoPathLabel;
        protected PathSelectionPanel pathSelectionPanel;
        protected JComponent settingsPathLabel;
        protected PathSelectionPanel settingsSelectionPanel;
        protected JRadioButton radioSkip;
        protected JRadioButton radioPath;
        protected String error;
        protected JComponent[]  dynamicComponents;

        /**
         * Create the panel.
         */
        public RepoCheckPanel(IzPanel parent, InstallData idata, boolean isOpaque)
        {
            int y = 0;
            String description = idata.langpack.getString("MavenRepoCheckPanel.info");
            String prompt = idata.langpack.getString("MavenRepoCheckPanel.prompt");
            String settingsinfo = idata.langpack.getString("MavenRepoCheckPanel.settingsInfo");
            String accept = idata.langpack.getString("MavenRepoCheckPanel.option1");
            String skip = idata.langpack.getString("MavenRepoCheckPanel.option2");
            
            
            GridBagLayout gridBagLayout = new GridBagLayout();
            gridBagLayout.columnWeights = new double[]{1.0};
            gridBagLayout.rowWeights = new double[]{0.0, 0.0, 0.0, 0.0, 0.0, 0.0};
            
            setLayout(gridBagLayout);
            setOpaque(isOpaque);
            
            JTextArea lblDescription = LabelFactory.createMultilineLabel(description, isOpaque);       
            GridBagConstraints gbc_lblDescription = new GridBagConstraints();
            gbc_lblDescription.anchor = GridBagConstraints.NORTHWEST;
            gbc_lblDescription.insets = new Insets(0, 0, 10, 0);
            gbc_lblDescription.fill = GridBagConstraints.BOTH;
            gbc_lblDescription.gridx = 0;
            gbc_lblDescription.gridy = y++;
            add(lblDescription, gbc_lblDescription);
            
            radioSkip = new JRadioButton(skip);
            radioSkip.setOpaque(isOpaque);
            radioSkip.setVerticalAlignment(SwingConstants.TOP);
            radioSkip.setHorizontalAlignment(SwingConstants.LEFT);
            GridBagConstraints gbc_radioSkip = new GridBagConstraints();
            gbc_radioSkip.fill = GridBagConstraints.HORIZONTAL;
            gbc_radioSkip.anchor = GridBagConstraints.WEST;
            gbc_radioSkip.gridx = 0;
            gbc_radioSkip.gridy = y++;
            add(radioSkip, gbc_radioSkip);
            
            // i dislike radio buttons.
            radioPath = new JRadioButton(String.format("<html><p>%1$s</p></html>",accept));
            radioPath.setOpaque(isOpaque);
            GridBagConstraints gbc_radioPath = new GridBagConstraints();
            gbc_radioPath.fill = GridBagConstraints.HORIZONTAL;
            gbc_radioPath.anchor = GridBagConstraints.WEST;
            gbc_radioPath.gridx = 0;
            gbc_radioPath.gridy = y++;
            add(radioPath, gbc_radioPath);
            
            repoPathLabel = new JLabel(prompt);
            repoPathLabel.setOpaque(isOpaque);
            repoPathLabel.setVisible(false);
            GridBagConstraints gbc_repoPathLabel = new GridBagConstraints();
            gbc_repoPathLabel.anchor = GridBagConstraints.WEST;
            gbc_repoPathLabel.gridx = 0;
            gbc_repoPathLabel.gridy = y++;
            add(repoPathLabel, gbc_repoPathLabel);
            
            pathSelectionPanel = new PathSelectionPanel(parent, idata);
            pathSelectionPanel.setOpaque(isOpaque);
            pathSelectionPanel.setVisible(false);
            GridBagConstraints gbc_pathSelectionPanel = new GridBagConstraints();
            gbc_pathSelectionPanel.fill = GridBagConstraints.HORIZONTAL;
            gbc_pathSelectionPanel.anchor = GridBagConstraints.NORTHWEST;
            gbc_pathSelectionPanel.gridx = 0;
            gbc_pathSelectionPanel.gridy = y++;
            add(pathSelectionPanel, gbc_pathSelectionPanel);
            pathSelectionPanel.setPath(idata.getVariable("USER_HOME") + File.separator + ".m2");
            
            
            settingsPathLabel = new JLabel(settingsinfo);
            settingsPathLabel.setOpaque(isOpaque);
            settingsPathLabel.setVisible(false);
            GridBagConstraints gbc_settingsPathLabel = new GridBagConstraints();
            gbc_settingsPathLabel.anchor = GridBagConstraints.WEST;
            gbc_settingsPathLabel.gridx = 0;
            gbc_settingsPathLabel.gridy = y++;
            add(settingsPathLabel, gbc_settingsPathLabel);
            
            settingsSelectionPanel = new PathSelectionPanel(parent, idata);
            settingsSelectionPanel.setOpaque(isOpaque);
            settingsSelectionPanel.setVisible(false);

            GridBagConstraints gbc_settingsSelectionPanel = new GridBagConstraints();
            gbc_settingsSelectionPanel.anchor = GridBagConstraints.WEST;
            gbc_settingsSelectionPanel.fill = GridBagConstraints.HORIZONTAL;
            gbc_settingsSelectionPanel.gridx = 0;
            gbc_settingsSelectionPanel.gridy = y++;
            add(settingsSelectionPanel, gbc_settingsSelectionPanel);

            settingsSelectionPanel.setPath(idata.getVariable("USER_HOME") + File.separator + ".m2" + File.separator + "settings.xml");
            settingsSelectionPanel.setAllFiles(true);

            JComponent [] dynamicComponents = 
            {
	    		repoPathLabel,
	        	pathSelectionPanel,
	        	settingsPathLabel,
	        	settingsSelectionPanel
	        };
            setDynamicComponents(dynamicComponents);
            setupRadioButtons();
            setAccessibilityData();

        }

        
        private void setAccessibilityData()
        {
            /*
             * AccessibleContext and ActionCommand setting for marathon automated testing
             */
            radioSkip.setActionCommand("Skip maven repository setup");
            AccessibleContext ac = radioSkip.getAccessibleContext();
            ac.setAccessibleDescription("This JRadioButton, if selected, will skip maven repository setup.");
            radioPath.setActionCommand("Install maven repository");
            ac = radioPath.getAccessibleContext();
            ac.setAccessibleDescription("This JRadioButton, if selected, will allow the user to specify a maven repository and settings.xml location");
            settingsSelectionPanel.getPathInputField().setActionCommand("Contains settings.xml location");
            ac = settingsSelectionPanel.getPathInputField().getAccessibleContext();
            ac.setAccessibleDescription("This JTextField contains the path to the user's settings.xml.");
	        pathSelectionPanel.getPathInputField().setActionCommand("Contains maven repository path");
	        ac = pathSelectionPanel.getPathInputField().getAccessibleContext();
            ac.setAccessibleDescription("This JTextField contains the path to the maven repository. Can be local or a URL.");
        }


        private void setupRadioButtons()
        {
            String stored = idata.getVariable("MAVEN_REPO_PATH");
            boolean prevPath = stored != null;
            ButtonGroup group = new ButtonGroup();
            group.add(radioPath);
            group.add(radioSkip);
            radioSkip.setSelected(!prevPath);
            radioPath.setSelected(prevPath);
            radioPath.addChangeListener(this);
            radioSkip.addChangeListener(this);
        }

       
        public JComponent getSettingsPathLabel()
        {
            return settingsPathLabel;
        }

        
        public PathSelectionPanel getSettingsSelectionPanel()
        {
            return settingsSelectionPanel;
        }
        
        public void stateChanged(ChangeEvent arg0)
        {
            AbstractButton abstractButton = (AbstractButton) arg0.getSource();
            ButtonModel buttonModel = abstractButton.getModel();
            if (buttonModel.isPressed() && buttonModel.isSelected()) {
                if (abstractButton == radioPath) 
                {
                	for (JComponent component : dynamicComponents)
                	{
                		component.setVisible(true);
                	}
                } 
                else 
                {
                	for (JComponent component : dynamicComponents)
                	{
                		component.setVisible(false);
                	}
                }
            }
        }
        
        public boolean validated()
        {
            if (radioSkip.isSelected()) {
                int answer = askQuestion("",idata.langpack.getString("MavenRepoCheckPanel.warning"), AbstractUIHandler.CHOICES_YES_NO, AbstractUIHandler.ANSWER_NO);
                if (answer != AbstractUIHandler.ANSWER_YES) {
                    return false;
                } 
                // clear out the variables
                idata.setVariable("mavenSettings","off");
            } else {
                
                String urlType = "";
                
                if (Arrays.asList(dynamicComponents).contains(pathSelectionPanel));
                {
	                String currentPath = pathSelectionPanel.getPath();
	                boolean isUrl = currentPath.toLowerCase().startsWith("http");
	                if (isUrl) 
	                {
	                    if (!IoHelper.remoteFileExists(currentPath))
	                    {
	                        emitError("Error", idata.langpack.getString("MavenRepoCheckPanel.path.error"));
	                        return false;
	                    }
	                } 
	                else 
	                {
	                    File chosenPath = new File(pathSelectionPanel.getPath());
	                    if (!chosenPath.exists() || !chosenPath.isDirectory())
	                    {
	                        emitError("Error", idata.langpack.getString("MavenRepoCheckPanel.path.error"));
	                        return false;
	                    }
	                    urlType = "file://";
	                }
                }
                
                File setFile = new File(settingsSelectionPanel.getPath());
                if (setFile.getPath().toLowerCase().startsWith("http")){ // little bit of a hacky check
                    emitError("Error", idata.langpack.getString("MavenRepoCheckPanel.settings.invalid"));
                    return false;
                }

                /**
                 * Check that we have an actual file selected and not just a dir.
                 */
                if (setFile.isDirectory())
                {
                    emitError("Error",
                            idata.langpack
                                    .getString("MavenRepoCheckPanel.settings.invalid.not.file"));
                    return false;
                }

                if (!setFile.exists()) // indicated settings location doesn't exist. warn about manual intervention.
                { 
                	int answer = askQuestion("", idata.langpack.getString("MavenRepoCheckPanel.settings.error"), 
                			AbstractUIHandler.CHOICES_YES_NO, AbstractUIHandler.ANSWER_NO);
                    if (answer != AbstractUIHandler.ANSWER_YES) // if the answer is yes, we save the path and make sure that we create the file with the same name in the Process    
                    {
                    	return false;
                    }             
                } 
                else 
                {
                    // maybe validate against the xsd?
                }
                idata.setVariable("mavenSettings", "on");
                idata.setVariable("MAVEN_SETTINGS_FULLPATH", setFile.getPath());
                if (Arrays.asList(dynamicComponents).contains(pathSelectionPanel));
                {
                	idata.setVariable("MAVEN_REPO_PATH", urlType + pathSelectionPanel.getPath());
                }
            }
            return true;
        }
        
        public String getError()
        {
            return error;
        }


        
        public void setError(String error)
        {
            this.error = error;
        }
        
        public void setDynamicComponents( JComponent[] dynamicComponents)
        {
        	this.dynamicComponents = dynamicComponents;
        }
        
    }
    
}
