/**
 *  Copyright 2005-2016 Red Hat, Inc.
 *
 *  Red Hat licenses this file to you under the Apache License, version
 *  2.0 (the "License"); you may not use this file except in compliance
 *  with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 *  implied.  See the License for the specific language governing
 *  permissions and limitations under the License.
 */
package io.fabric8.boot.commands;

import io.fabric8.api.*;
import io.fabric8.boot.commands.support.ResolverPolicyEnum;
import io.fabric8.utils.OsgiUtils;
import io.fabric8.utils.PasswordEncoder;
import io.fabric8.utils.Ports;
import io.fabric8.utils.shell.ShellUtils;
import io.fabric8.zookeeper.bootstrap.BootstrapConfiguration;

import java.io.IOException;
import java.nio.file.Path;
import java.security.AccessControlContext;
import java.security.AccessController;
import java.security.Principal;
import java.util.Arrays;
import java.util.Dictionary;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;

import javax.security.auth.Subject;

import org.apache.felix.gogo.commands.Argument;
import org.apache.felix.gogo.commands.Command;
import org.apache.felix.gogo.commands.Option;
import org.apache.felix.utils.properties.Properties;
import org.apache.karaf.jaas.modules.BackingEngine;
import org.apache.karaf.shell.console.AbstractAction;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceReference;
import org.osgi.service.cm.Configuration;
import org.osgi.service.cm.ConfigurationAdmin;

import com.google.common.base.Strings;

@Command(name = "create", scope = "fabric", description = "Creates a new fabric ensemble (ZooKeeper ensemble)", detailedDescription = "classpath:create.txt")
class CreateAction extends AbstractAction {

    private static final String GIT_REMOTE_USER = "gitRemoteUser";
    private static final String GIT_REMOTE_PASSWORD = "gitRemotePassword";

    @Option(name = "--clean", description = "Clean local zookeeper cluster and configurations")
    private boolean clean;
    @Option(name = "--no-import", description = "Disable the import of the sample registry data")
    private boolean noImport;
    @Option(name = "--import-dir", description = "Directory of files to import into the newly created ensemble")
    private String importDir;
    @Option(name = "-v", aliases = {"--verbose"}, description = "Flag to enable verbose output of files being imported")
    boolean verbose = false;
    @Option(name = "-f", aliases = "--force", multiValued = false, description = "Forces re-creating fabric")
    private boolean force;
    @Option(name = "-g", aliases = {"--global-resolver"}, description = "The global resolver policy, which becomes the default resolver policy applied to all new containers created in this fabric. Possible values are: localip, localhostname, publicip, publichostname, manualip. Default is localhostname.")
    String globalResolver;
    @Option(name = "-r", aliases = {"--resolver"}, description = "The local resolver policy. Possible values are: localip, localhostname, publicip, publichostname, manualip. Default is localhostname.")
    String resolver;
    @Option(name = "-m", aliases = {"--manual-ip"}, description = "An address to use, when using the manualip resolver.")
    String manualIp;
    @Option(name = "-b", aliases = {"--bind-address"}, description = "The default bind address.")
    String bindAddress;
    @Option(name = "-n", aliases = "--non-managed", multiValued = false, description = "Flag to keep the container non managed")
    private boolean nonManaged;
    @Option(name = "--wait-for-provisioning", multiValued = false, description = "Flag to wait for the initial container provisioning")
    private boolean waitForProvisioning=false;
    @Option(name = "--bootstrap-timeout", multiValued = false, description = "How long to wait (milliseconds) for the initial fabric bootstrap")
    private long bootstrapTimeout =120000L;
    @Option(name = "-t", aliases = {"--time"}, description = "How long to wait (milliseconds) for the ensemble to start up before trying to import the default data")
    long ensembleStartupTime = 2000L;
    @Option(name = "-p", aliases = "--profile", multiValued = true, description = "Chooses the profile of the container. To specify multiple profiles, use this flag multiple times.")
    private Set<String> profiles = null;
    @Option(name = "-v", aliases = "--version", multiValued = false, description = "Chooses the default version.")
    private String version = ContainerOptions.DEFAULT_VERSION;
    @Option(name = "--min-port", multiValued = false, description = "The minimum port of the allowed port range")
    private int minimumPort = Ports.MIN_PORT_NUMBER;
    @Option(name = "--max-port", multiValued = false, description = "The maximum port of the allowed port range")
    private int maximumPort = Ports.MAX_PORT_NUMBER;
    @Option(name = "--zookeeper-ticktime", multiValued = false, description = "The length of a single tick, which is the basic time unit used by ZooKeeper, as measured in milliseconds. It is used to regulate heartbeats, and timeouts. For example, the minimum session timeout will be two ticks")
    private int zooKeeperTickTime = CreateEnsembleOptions.DEFAULT_TICKTIME;
    @Option(name = "--zookeeper-init-limit", multiValued = false, description = "The amount of time, in ticks (see tickTime), to allow followers to connect and sync to a leader")
    private int zooKeeperInitLimit = CreateEnsembleOptions.DEFAULT_INIT_LIMIT;
    @Option(name = "--zookeeper-sync-limit", multiValued = false, description = "The amount of time, in ticks (see tickTime), to allow followers to sync with ZooKeeper")
    private int zooKeeperSyncLimit = CreateEnsembleOptions.DEFAULT_SYNC_LIMIT;
    @Option(name = "--zookeeper-data-dir", multiValued = false, description = "The location where ZooKeeper will store the in-memory database snapshots and, unless specified otherwise, the transaction log of updates to the database.")
    private String zooKeeperDataDir = CreateEnsembleOptions.DEFAULT_DATA_DIR;
    @Option(name = "--zookeeper-password", multiValued = false, description = "The ensemble password to use (one will be generated if not given)")
    private String zookeeperPassword;
    @Option(name = "--zookeeper-snap-retain-count", multiValued = false, description = "Number of snapshots to be retained after purge in Zookeeper")
    private int zookeeperSnapRetainCount = CreateEnsembleOptions.DEFAULT_SNAP_RETAIN_COUNT;
    @Option(name = "--zookeeper-purge-interval", multiValued = false, description = "Zookeeper snapshots purge interval in hours")
    private int zookeeperPurgeInterval = CreateEnsembleOptions.DEFAULT_PURGE_INTERVAL_IN_HOURS;
    @Option(name = "--zookeeper-server-port", multiValued = false, description = "The main port for ZooKeeper server")
    private int zooKeeperServerPort = -1;
    @Option(name = "--generate-zookeeper-password", multiValued = false, description = "Flag to enable automatic generation of password")
    private boolean generateZookeeperPassword = false;
    @Option(name = "--new-user", multiValued = false, description = "The username of a new user. The option refers to karaf user (ssh, http, jmx).")
    private String newUser;
    @Option(name = "--new-user-password", multiValued = false, description = "The password of the new user. The option refers to karaf user (ssh, http, jmx).")
    private String newUserPassword;
    @Option(name = "--external-git-url", multiValued = false, description = "Specify an external git URL.")
    private String externalGitUrl;
    @Option(name = "--external-git-user", multiValued = false, description = "Specify an external git user.")
    private String externalGitUser;
    @Option(name = "--external-git-password", multiValued = false, description = "Specify an external git password.")
    private String externalGitPassword;
    @Option(name = "--new-user-role", multiValued = false, description = "The role of the new user. The option refers to karaf user (ssh, http, jmx).")
    private String newUserRole = "";

    @Argument(required = false, multiValued = true, description = "List of containers. Empty list assumes current container only.")
    private List<String> containers;

    private static final String ROLE_DELIMITER = ",";
    
    private final BundleContext bundleContext;
    private final ConfigurationAdmin configAdmin;
    private final ZooKeeperClusterBootstrap bootstrap;
    private final RuntimeProperties runtimeProperties;

    CreateAction(BundleContext bundleContext, ConfigurationAdmin configAdmin, ZooKeeperClusterBootstrap bootstrap, RuntimeProperties runtimeProperties) {
        this.bundleContext = bundleContext;
        this.configAdmin = configAdmin;
        this.bootstrap = bootstrap;
        this.runtimeProperties = runtimeProperties;

        Path homePath = runtimeProperties.getHomePath();
        importDir = homePath.resolve("fabric").resolve("import").toFile().getAbsolutePath();
    }

    protected Object doExecute() throws Exception {

    	String adminRole = "admin";
    	String administratorRole = "administrator";
    	String superUserRole = "superuser";
        boolean adminExists = hasRole(adminRole, superUserRole, administratorRole);
    	if (!adminExists) {
    		System.out.println("The fabric:create command can be executed only by admin user");
    		return null;
    	}
    	
        Path propsPath = runtimeProperties.getConfPath().resolve("users.properties");
        Properties userProps = new Properties(propsPath.toFile());
        
    	if (!adminExists && !usersPropertiesFileContainsRole(userProps, adminRole, superUserRole, administratorRole)) {
    		System.out.println("The etc/user.properties file must contain at least one admin user, if no other admin exists");
    		return null;
    	}
    	
        // prevent creating fabric if already created
        ServiceReference<FabricService> sref = bundleContext.getServiceReference(FabricService.class);
        FabricService fabricService = sref != null ? bundleContext.getService(sref) : null;
        if (!force && (fabricService != null && fabricService.getCurrentContainer().isEnsembleServer())) {
            System.out.println("Current container " + fabricService.getCurrentContainerName() + " is already in the current fabric ensemble. Cannot create fabric.");
            System.out.println("You can use the --force option, if you want to force re-create the fabric.");
            return null;
        }

        Configuration bootConfiguration = configAdmin.getConfiguration(BootstrapConfiguration.COMPONENT_PID, null);
        Dictionary<String, Object> bootProperties = bootConfiguration.getProperties();
        if (bootProperties == null) {
            bootProperties = new Hashtable<>();
        }

        String runtimeIdentity = runtimeProperties.getRuntimeIdentity();
        CreateEnsembleOptions.Builder<?> builder = CreateEnsembleOptions.builder()
                .zooKeeperServerTickTime(zooKeeperTickTime)
                .zooKeeperServerInitLimit(zooKeeperInitLimit)
                .zooKeeperServerSyncLimit(zooKeeperSyncLimit)
                .zooKeeperServerDataDir(zooKeeperDataDir)
                .zookeeperSnapRetainCount(zookeeperSnapRetainCount)
                .zookeeperPurgeInterval(zookeeperPurgeInterval)
                .fromRuntimeProperties(runtimeProperties)
                .bootstrapTimeout(bootstrapTimeout)
                .waitForProvision(waitForProvisioning)
                .autoImportEnabled(!noImport)
                .clean(clean);

        builder.version(version);

        if (containers == null || containers.isEmpty()) {
            containers = Arrays.asList(runtimeIdentity);
        }

        if (!noImport && importDir != null) {
            builder.autoImportEnabled(true);
            builder.importPath(importDir);
        }

        if (globalResolver != null) {
            if (!isInEnum(globalResolver, ResolverPolicyEnum.class)) {
                System.out.println("The globalResolver value must be one of the following: localip, localhostname, publicip, publichostname, manualip");
                return null;
            }
            builder.globalResolver(globalResolver);
            bootProperties.put(ZkDefs.GLOBAL_RESOLVER_PROPERTY, globalResolver);
        }

        if (resolver != null) {
            if (!isInEnum(resolver, ResolverPolicyEnum.class)) {
                System.out.println("The resolver value must be one of the following: localip, localhostname, publicip, publichostname, manualip");
                return null;
            }
            builder.resolver(resolver);
            bootProperties.put(ZkDefs.LOCAL_RESOLVER_PROPERTY, resolver);
        }

        if (manualIp != null) {
            builder.manualIp(manualIp);
            bootProperties.put(ZkDefs.MANUAL_IP, manualIp);
        }

        if (bindAddress != null) {
            if (!bindAddress.contains(":")) {
                builder.bindAddress(bindAddress);
                bootProperties.put(ZkDefs.BIND_ADDRESS, bindAddress);
            } else {
                String[] parts = bindAddress.split(":");
                builder.bindAddress(parts[0]);
                builder.zooKeeperServerPort(Integer.parseInt(parts[1]));
                bootProperties.put(ZkDefs.BIND_ADDRESS, parts[0]);
            }
        }

        if (zooKeeperServerPort > 0) {
            // --zookeeper-server-port option has higher priority than
            // CreateEnsembleOptions.ZOOKEEPER_SERVER_PORT and CreateEnsembleOptions.ZOOKEEPER_SERVER_CONNECTION_PORT
            // system/runtime properties
            builder.setZooKeeperServerPort(zooKeeperServerPort);
            builder.setZooKeeperServerConnectionPort(zooKeeperServerPort);
        } else {
            int shiftedPort = Ports.mapPortToRange(2181, minimumPort, maximumPort);
            builder.setZooKeeperServerPort(shiftedPort);
            builder.setZooKeeperServerConnectionPort(shiftedPort);
        }

        //Configure External Git Repository.
        if (externalGitUrl != null) {
            builder.dataStoreProperty(Constants.GIT_REMOTE_URL, externalGitUrl);
        }
        if (externalGitUser != null) {
            builder.dataStoreProperty(GIT_REMOTE_USER, externalGitUser);
        }
        if (externalGitPassword != null) {
            builder.dataStoreProperty(GIT_REMOTE_PASSWORD, externalGitPassword);
        }

        if ((externalGitUrl != null) || (externalGitUser != null) || (externalGitPassword != null)) {
            Configuration configuration = configAdmin.getConfiguration(Constants.DATASTORE_PID);
            Dictionary<String, Object> existingProperties = configuration.getProperties();
            Map<String, String> changedProperties = builder.getDataStoreProperties();
            for (String key : changedProperties.keySet()) {
                existingProperties.put(key, changedProperties.get(key));
            }
            configuration.update(existingProperties);
        }

        if (profiles != null && profiles.size() > 0) {
            builder.profiles(profiles);
        }

        if (nonManaged) {
            builder.agentEnabled(false);
        } else {
            builder.agentEnabled(true);
        }

        builder.minimumPort(minimumPort);
        builder.maximumPort(maximumPort);
        bootProperties.put(ZkDefs.MINIMUM_PORT, String.valueOf(minimumPort));
        bootProperties.put(ZkDefs.MAXIMUM_PORT, String.valueOf(maximumPort));

        newUser = newUser != null ? newUser : ShellUtils.retrieveFabricUser(session);
        newUserPassword = newUserPassword != null ? newUserPassword : ShellUtils.retrieveFabricUserPassword(session);

        if (userProps.isEmpty()) {
            String[] credentials = promptForNewUser(newUser, newUserPassword);
            newUser = credentials[0];
            newUserPassword = credentials[1];
        } else {
            if (newUser == null || newUserPassword == null) {
                newUser = "" + userProps.keySet().iterator().next();
                newUserPassword = "" + userProps.get(newUser);
                if (newUserPassword.contains(ROLE_DELIMITER)) {
                    newUserPassword = newUserPassword.substring(0, newUserPassword.indexOf(ROLE_DELIMITER));
                }
            }
            String passwordWithroles = userProps.get(newUser);
            if (passwordWithroles != null && passwordWithroles.contains(ROLE_DELIMITER)) {
                String[] infos = passwordWithroles.split(",");
                String oldUserRole = "";
                if (newUserIsAdmin(infos)) {
                    newUserRole = "_g_:admin";
                    oldUserRole = newUserRole;
                }
                for (int i = 1; i < infos.length; i++) {
                    if (infos[i].trim().startsWith(BackingEngine.GROUP_PREFIX)) {
                        // it's a group reference
                        String groupInfo = (String) userProps.get(infos[i].trim());
                        if (groupInfo != null) {
                            String[] roles = groupInfo.split(",");
                            for (int j = 1; j < roles.length; j++) {
                                if (!roles[j].trim().equals(oldUserRole)) {
                                    if (!newUserRole.isEmpty()) {
                                        newUserRole = newUserRole + ROLE_DELIMITER + roles[j].trim();
                                    } else {
                                        newUserRole = roles[j].trim();
                                    }
                                }
                            }
                        }
                    } else {
                        // it's an user reference
                        if (!infos[i].trim().equals(oldUserRole)) {
                            if (!newUserRole.isEmpty()) {
                                newUserRole = newUserRole + ROLE_DELIMITER + infos[i].trim();
                            } else {
                                newUserRole = infos[i].trim();
                            }
                        }
                    }                
                }
            }
        }

        if (Strings.isNullOrEmpty(newUser)) {
            System.out.println("No user specified. Cannot create a new fabric ensemble.");
            return null;
        }

        StringBuilder sb = new StringBuilder();

        // session is unset when this is called from FMC
        if (session != null) {
            ShellUtils.storeFabricCredentials(session, newUser, newUserPassword);
        }

        if (generateZookeeperPassword) {
            //do nothing use the generated password.
        } else if (zookeeperPassword == null) {
            zookeeperPassword = PasswordEncoder.decode(runtimeProperties.getProperty(CreateEnsembleOptions.ZOOKEEPER_PASSWORD, PasswordEncoder.encode(newUserPassword)));
            builder.zookeeperPassword(zookeeperPassword);
        } else {
            builder.zookeeperPassword(zookeeperPassword);
        }

        OsgiUtils.updateCmConfigurationAndWait(bundleContext, bootConfiguration, bootProperties, bootstrapTimeout, TimeUnit.MILLISECONDS);

        String roleToAssign = newUserRole.isEmpty() ? "_g_:admin":newUserRole;
        
        CreateEnsembleOptions options = builder.users(userProps)
                                               .withUser(newUser, newUserPassword , roleToAssign)
                                               .build();

        if (containers.size() == 1 && containers.contains(runtimeIdentity)) {
            bootstrap.create(options);
        } else {
            ServiceProxy<ZooKeeperClusterService> serviceProxy = ServiceProxy.createServiceProxy(bundleContext, ZooKeeperClusterService.class);
            try {
                serviceProxy.getService().createCluster(containers, options);
            } finally {
                serviceProxy.close();
            }
        }

        ShellUtils.storeZookeeperPassword(session, options.getZookeeperPassword());
        if (zookeeperPassword == null && !generateZookeeperPassword) {
            sb.append("Zookeeper password: (reusing users ").append(newUser).append(" password:").append(options.getZookeeperPassword()).append(")\n");
            sb.append("(You can use the --zookeeper-password / --generate-zookeeper-password option to specify one.)\n");
        }   else if (generateZookeeperPassword) {
            sb.append("Generated zookeeper password:").append(options.getZookeeperPassword());
        }
        System.out.println(sb.toString());
        if (!nonManaged && !waitForProvisioning) {
            System.out.println("It may take a couple of seconds for the container to provision...");
            System.out.println("You can use the --wait-for-provisioning option, if you want this command to block until the container is provisioned.");
        }

        return null;
    }

    private String[] promptForNewUser(String user, String password) throws IOException {
        String[] response = new String[2];
        // If the username was not configured via cli, then prompt the user for the values
        if (user == null || password == null) {
            System.out.println("No user found in etc/users.properties or specified as an option. Please specify one ...");
        }
        while (user == null || user.isEmpty()) {
            user = ShellUtils.readLine(session, "New user name: ", false);
            if (user == null) {
                break;
            }
        }

        if (user != null && password == null) {
            String password1 = null;
            String password2 = null;
            while (password1 == null || !password1.equals(password2)) {
                password1 = ShellUtils.readLine(session, "Password for " + user + ": ", true);
                password2 = ShellUtils.readLine(session, "Verify password for " + user + ": ", true);

                if (password1 == null || password2 == null) {
                    break;
                }

                if (password1 != null && password1.equals(password2)) {
                    password = password1;
                } else {
                    System.out.println("Passwords did not match. Please try again!");
                }
            }
        }
        response[0] = user;
        response[1] = password;
        return response;
    }
    
    private boolean hasRole(String... requestedRole) {
    	AccessControlContext acc = AccessController.getContext();
    	if (acc == null) {
    	    return false;
    	}
    	Subject subject = Subject.getSubject(acc);
    	if (subject == null) {
    	    return false;
    	}
    	return currentUserHasRole(subject.getPrincipals(), requestedRole);
    }
    
    private boolean currentUserHasRole(Set<Principal> principals, String... requestedRole) {
    	boolean hasRole = false;
    	Iterator<Principal> it = principals.iterator();
    	while (it.hasNext()) {
			Principal principal = (Principal) it.next();
			for (int i = 0; i < requestedRole.length; i++) {
			    if (principal.getName().equalsIgnoreCase(requestedRole[i])) hasRole = true;
			}
		}
    	return hasRole;
    }
    
    private boolean newUserIsAdmin(String... roles) {
        boolean newUserIsAdmin = false;
        for (int i = 0; i < roles.length; i++) {
            if (roles[i].equalsIgnoreCase("admin")) newUserIsAdmin = true;
        }
        return newUserIsAdmin;
    }
    
    private boolean usersPropertiesFileContainsRole(Properties userProps, String... requestedRole) {
    	boolean adminExists = false;
    	Iterator<String> usersIterator = userProps.keySet().iterator();
    	while (usersIterator.hasNext()) {
			String key = (String) usersIterator.next();
			String passwordWithroles = userProps.get(key);
			String[] infos = passwordWithroles.split(",");
            for (int i = 1; i < infos.length; i++) {
                if (infos[i].trim().startsWith(BackingEngine.GROUP_PREFIX)) {
                    String groupInfo = (String) userProps.get(infos[i].trim());
                    if (groupInfo != null) {
                        String[] roles = groupInfo.split(",");
                        for (int j = 1; j < roles.length; j++) {
                        	for (int z = 0; z < requestedRole.length; z++) {
                                if (roles[j].trim().equalsIgnoreCase(requestedRole[z])) {
                                    adminExists = true;
                                }
                        	}
                        }
                    }
                } else {
                	for (int z = 0; z < requestedRole.length; z++) {
                        if (infos[i].trim().equalsIgnoreCase(requestedRole[z])) {
                            adminExists = true;
                        }
                	}
                }                
            }
		}
    	return adminExists;
    }
    
    private <E extends Enum<E>> boolean isInEnum(String value, Class<E> enumClass) {
        for (E e : enumClass.getEnumConstants()) {
            if(((ResolverPolicyEnum) e).getResolver().equals(value)) { 
                return true; 
            }
        }
       return false;
    }

    public String getBindAddress() {
        return bindAddress;
    }

    public void setBindAddress(String bindAddress) {
        this.bindAddress = bindAddress;
    }

    public boolean isClean() {
        return clean;
    }

    public void setClean(boolean clean) {
        this.clean = clean;
    }

    public boolean isNoImport() {
        return noImport;
    }

    public void setNoImport(boolean noImport) {
        this.noImport = noImport;
    }

    public String getImportDir() {
        return importDir;
    }

    public void setImportDir(String importDir) {
        this.importDir = importDir;
    }

    public boolean isVerbose() {
        return verbose;
    }

    public void setVerbose(boolean verbose) {
        this.verbose = verbose;
    }

    public long getEnsembleStartupTime() {
        return ensembleStartupTime;
    }

    public void setEnsembleStartupTime(long ensembleStartupTime) {
        this.ensembleStartupTime = ensembleStartupTime;
    }

    public List<String> getContainers() {
        return containers;
    }

    public void setContainers(List<String> containers) {
        this.containers = containers;
    }

    public int getMinimumPort() {
        return minimumPort;
    }

    public void setMinimumPort(int minimumPort) {
        this.minimumPort = minimumPort;
    }

    public int getMaximumPort() {
        return maximumPort;
    }

    public void setMaximumPort(int maximumPort) {
        this.maximumPort = maximumPort;
    }

    public String getZookeeperPassword() {
        return zookeeperPassword;
    }

    public void setZookeeperPassword(String zookeeperPassword) {
        this.zookeeperPassword = zookeeperPassword;
    }

    public String getNewUser() {
        return newUser;
    }

    public void setNewUser(String newUser) {
        this.newUser = newUser;
    }

    public String getNewUserPassword() {
        return newUserPassword;
    }

    public void setNewUserPassword(String newUserPassword) {
        this.newUserPassword = newUserPassword;
    }

    public String getNewUserRole() {
        return newUserRole;
    }

    public void setNewUserRole(String newUserRole) {
        this.newUserRole = newUserRole;
    }

    public Set<String> getProfiles() {
        return profiles;
    }

    public void setProfiles(Set<String> profiles) {
        this.profiles = profiles;
    }

    public boolean isNonManaged() {
        return nonManaged;
    }

    public void setNonManaged(boolean nonManaged) {
        this.nonManaged = nonManaged;
    }

    public String getGlobalResolver() {
        return globalResolver;
    }

    public void setGlobalResolver(String globalResolver) {
        this.globalResolver = globalResolver;
    }

    public String getResolver() {
        return resolver;
    }

    public void setResolver(String resolver) {
        this.resolver = resolver;
    }

    public String getManualIp() {
        return manualIp;
    }

    public void setManualIp(String manualIp) {
        this.manualIp = manualIp;
    }

    public boolean isGenerateZookeeperPassword() {
        return generateZookeeperPassword;
    }

    public void setGenerateZookeeperPassword(boolean generateZookeeperPassword) {
        this.generateZookeeperPassword = generateZookeeperPassword;
    }
}
