package io.hawt.wildfly;

import org.wildfly.security.auth.principal.NamePrincipal;
import org.wildfly.security.auth.server.SecurityIdentity;
import org.wildfly.security.authz.Roles;
import org.wildfly.security.credential.*;
import org.wildfly.security.manager.WildFlySecurityManager;

import javax.security.auth.Subject;
import java.security.AccessController;
import java.security.Principal;
import java.security.PrivilegedAction;
import java.util.Set;

/* Copied from org.jboss.as.webservices.util.SubjectUtil because this was compiled with Java 11 and Fuse 7.12 requires
to be compilable by Java 8 */
public class SubjectUtil {

    public static Subject fromSecurityIdentity(SecurityIdentity securityIdentity, Subject subject) {
        if (subject == null) {
            subject = new Subject();
        }
        // The first principal added must be the security identity principal
        // as logic in both CXF and JBoss WS look for the first non-Group principal
        subject.getPrincipals().add(securityIdentity.getPrincipal());

        Roles identityRoles = securityIdentity.getRoles();
        // Just add a simple principal for each role instead of aggregating them in a Group.
        // CXF can use such principals when identifying the subject's roles
        String principalName = securityIdentity.getPrincipal().getName();
        Set<Principal> principals = subject.getPrincipals();
        for (String role : identityRoles) {
            if (!principalName.equals(role)) {
                principals.add(new NamePrincipal(role));
            }
        }

        // Don't bother with the 'CallerPrincipal' group, since if there is no Group class,
        // legacy security realms that use that Group to find the 'caller principal' cannot
        // be in use

        // process the identity's public and private credentials.
        for (Credential credential : securityIdentity.getPublicCredentials()) {
            if (credential instanceof PublicKeyCredential) {
                subject.getPublicCredentials().add(credential.castAs(PublicKeyCredential.class).getPublicKey());
            }
            else if (credential instanceof X509CertificateChainPublicCredential) {
                subject.getPublicCredentials().add(credential.castAs(X509CertificateChainPublicCredential.class).getCertificateChain());
            }
            else {
                subject.getPublicCredentials().add(credential);
            }
        }

        for (Credential credential : securityIdentity.getPrivateCredentials()) {
            if (credential instanceof PasswordCredential) {
                addPrivateCredential(subject, credential.castAs(PasswordCredential.class).getPassword());
            }
            else if (credential instanceof SecretKeyCredential) {
                addPrivateCredential(subject, credential.castAs(SecretKeyCredential.class).getSecretKey());
            }
            else if (credential instanceof KeyPairCredential) {
                addPrivateCredential(subject, credential.castAs(KeyPairCredential.class).getKeyPair());
            }
            else if (credential instanceof X509CertificateChainPrivateCredential) {
                addPrivateCredential(subject, credential.castAs(X509CertificateChainPrivateCredential.class).getCertificateChain());
            }
            else {
                addPrivateCredential(subject, credential);
            }
        }

        // add the identity itself as a private credential - integration code can interact with the SI instead of the Subject if desired.
        addPrivateCredential(subject, securityIdentity);

        return subject;
    }

    private static void addPrivateCredential(Subject subject, Object credential) {
        if (!WildFlySecurityManager.isChecking()) {
            subject.getPrivateCredentials().add(credential);
        } else {
            AccessController.doPrivileged((PrivilegedAction<Object>) () -> {
                subject.getPrivateCredentials().add(credential);
                return null;
            });
        }

    }

}
