/*
 * Copyright 2019 Red Hat
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.apicurio.tests.smokeTests.apicurio;

import io.apicurio.registry.rest.beans.ArtifactMetaData;
import io.apicurio.registry.rest.beans.Rule;
import io.apicurio.registry.rest.beans.UpdateState;
import io.apicurio.registry.rest.beans.VersionMetaData;
import io.apicurio.registry.types.ArtifactState;
import io.apicurio.registry.types.ArtifactType;
import io.apicurio.registry.types.RuleType;
import io.apicurio.registry.utils.ConcurrentUtil;
import io.apicurio.registry.utils.IoUtil;
import io.apicurio.tests.BaseIT;
import io.apicurio.tests.utils.subUtils.ArtifactUtils;
import io.vertx.core.json.JsonObject;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Tag;
import org.junit.jupiter.api.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static io.apicurio.tests.Constants.SMOKE;
import static org.hamcrest.CoreMatchers.hasItems;
import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.not;
import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertThrows;
import static org.junit.jupiter.api.Assertions.fail;

import java.io.ByteArrayInputStream;
import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletionStage;
import javax.ws.rs.WebApplicationException;

@Tag(SMOKE)
class ArtifactsIT extends BaseIT {

    private static final Logger LOGGER = LoggerFactory.getLogger(ArtifactsIT.class);

    @Test
    void createAndUpdateArtifact() {
        Rule rule = new Rule();
        rule.setType(RuleType.VALIDITY);
        rule.setConfig("FULL");

        LOGGER.info("Creating global rule:{}", rule.toString());
        apicurioService.createGlobalRule(rule);

        String artifactId = "createAndUpdateArtifactId1";

        ByteArrayInputStream artifactData = new ByteArrayInputStream("{\"type\":\"record\",\"name\":\"myrecord1\",\"fields\":[{\"name\":\"foo\",\"type\":\"string\"}]}".getBytes(StandardCharsets.UTF_8));
        ArtifactMetaData metaData = ArtifactUtils.createArtifact(apicurioService, ArtifactType.AVRO, artifactId, artifactData);
        LOGGER.info("Created artifact {} with metadata {}", artifactId, metaData.toString());

        JsonObject response = new JsonObject(apicurioService.getLatestArtifact(artifactId).readEntity(String.class));

        LOGGER.info("Artifact with name:{} and content:{} was created", response.getString("name"), response.toString());

        String invalidArtifactDefinition = "<type>record</type>\n<name>test</name>";
        artifactData = new ByteArrayInputStream(invalidArtifactDefinition.getBytes(StandardCharsets.UTF_8));
        String invalidArtifactId = "createAndUpdateArtifactId2";

        try {
            LOGGER.info("Invalid artifact sent {}", invalidArtifactDefinition);
            ArtifactUtils.createArtifact(apicurioService, ArtifactType.AVRO, invalidArtifactId, artifactData);
        } catch (WebApplicationException e) {
            assertThat("{\"message\":\"Syntax violation for Avro artifact.\",\"error_code\":400}", is(e.getResponse().readEntity(String.class)));
        }

        artifactData = new ByteArrayInputStream("{\"type\":\"record\",\"name\":\"myrecord1\",\"fields\":[{\"name\":\"bar\",\"type\":\"long\"}]}".getBytes(StandardCharsets.UTF_8));
        metaData = ArtifactUtils.updateArtifact(apicurioService, ArtifactType.AVRO, artifactId, artifactData);
        LOGGER.info("Artifact with ID {} was updated: {}", artifactId, metaData.toString());

        response = new JsonObject(apicurioService.getLatestArtifact(artifactId).readEntity(String.class));

        LOGGER.info("Artifact with ID {} was updated: {}", artifactId, response.toString());

        List<Long> apicurioVersions = apicurioService.listArtifactVersions(artifactId);

        LOGGER.info("Available versions of artifact with ID {} are: {}", artifactId, apicurioVersions.toString());
        assertThat(apicurioVersions, hasItems(1L, 2L));

        response = new JsonObject(apicurioService.getArtifactVersion(1, artifactId).readEntity(String.class));

        LOGGER.info("Artifact with ID {} and version {}: {}", artifactId, 1, response.toString());

        assertThat(response.getJsonArray("fields").getJsonObject(0).getString("name"), is("foo"));
    }

    @Test
    void createAndDeleteMultipleArtifacts() {
        LOGGER.info("Creating some artifacts...");
        Map<String, String> idMap = createMultipleArtifacts(10);
        LOGGER.info("Created  {} artifacts", idMap.size());

        deleteMultipleArtifacts(idMap);

        for (Map.Entry entry : idMap.entrySet()) {
            try {
                apicurioService.getLatestArtifact(entry.getValue().toString());
            } catch (WebApplicationException e) {
                assertThat("{\"message\":\"No artifact with ID '" + entry.getValue() + "' was found.\",\"error_code\":404}", is(e.getResponse().readEntity(String.class)));
            }
        }
    }

    @Test
    void deleteArtifactSpecificVersion() {
        ByteArrayInputStream artifactData = new ByteArrayInputStream("{\"type\":\"record\",\"name\":\"myrecordx\",\"fields\":[{\"name\":\"foo\",\"type\":\"string\"}]}".getBytes(StandardCharsets.UTF_8));
        String artifactId = "deleteArtifactSpecificVersionId";
        ArtifactMetaData metaData = ArtifactUtils.createArtifact(apicurioService, ArtifactType.AVRO, artifactId, artifactData);
        LOGGER.info("Created artifact {} with metadata {}", artifactId, metaData.toString());

        JsonObject response = new JsonObject(apicurioService.getLatestArtifact(artifactId).readEntity(String.class));

        LOGGER.info("Created record with name:{} and content:{}", response.getString("name"), response.toString());

        for (int x = 0; x < 9; x++) {
            String artifactDefinition = "{\"type\":\"record\",\"name\":\"myrecordx\",\"fields\":[{\"name\":\"foo" + x + "\",\"type\":\"string\"}]}";
            artifactData = new ByteArrayInputStream(artifactDefinition.getBytes(StandardCharsets.UTF_8));
            metaData = ArtifactUtils.updateArtifact(apicurioService, ArtifactType.AVRO, artifactId, artifactData);
            LOGGER.info("Artifact with ID {} was updated: {}", artifactId, metaData.toString());
        }

        List<Long> artifactVersions = apicurioService.listArtifactVersions(artifactId);

        LOGGER.info("Available versions of artifact with ID {} are: {}", artifactId, artifactVersions.toString());
        assertThat(artifactVersions, hasItems(1L, 2L, 3L, 4L, 5L, 6L, 7L, 8L, 9L, 10L));

        apicurioService.deleteArtifactVersion(4, artifactId);
        LOGGER.info("Version 4 of artifact {} was deleted", artifactId);

        artifactVersions = apicurioService.listArtifactVersions(artifactId);

        LOGGER.info("Available versions of artifact with ID {} are: {}", artifactId, artifactVersions.toString());
        assertThat(artifactVersions, hasItems(1L, 2L, 3L, 5L, 6L, 7L, 8L, 9L, 10L));
        assertThat(artifactVersions, not(hasItems(4L)));

        try {
            apicurioService.getArtifactVersion(4, artifactId);
        } catch (WebApplicationException e) {
            assertThat("{\"message\":\"No version '4' found for artifact with ID 'deleteArtifactSpecificVersionId'.\",\"error_code\":404}", is(e.getResponse().readEntity(String.class)));
        }

        artifactData = new ByteArrayInputStream("{\"type\":\"record\",\"name\":\"myrecordx\",\"fields\":[{\"name\":\"foo11\",\"type\":\"string\"}]}".getBytes(StandardCharsets.UTF_8));
        metaData = ArtifactUtils.updateArtifact(apicurioService, ArtifactType.AVRO, artifactId, artifactData);
        LOGGER.info("Artifact with ID {} was updated: {}", artifactId, metaData.toString());

        artifactVersions = apicurioService.listArtifactVersions(artifactId);

        LOGGER.info("Available versions of artifact with ID {} are: {}", artifactId, artifactVersions.toString());

        assertThat(artifactVersions, hasItems(1L, 2L, 3L, 5L, 6L, 7L, 8L, 9L, 10L, 11L));
        assertThat(artifactVersions, not(hasItems(4L)));
    }

    @Test
    void createNonAvroArtifact() {
        ByteArrayInputStream artifactData = new ByteArrayInputStream("{\"type\":\"INVALID\",\"config\":\"invalid\"}".getBytes(StandardCharsets.UTF_8));
        String artifactId = "artifactWithNonAvroFormatId";

        CompletionStage<ArtifactMetaData> csResult = apicurioService.createArtifact(ArtifactType.JSON, artifactId, artifactData);
        ArtifactMetaData metaData = ConcurrentUtil.result(csResult);

        LOGGER.info("Created artifact {} with metadata {}", artifactId, metaData.toString());

        JsonObject response = new JsonObject(apicurioService.getLatestArtifact(artifactId).readEntity(String.class));

        LOGGER.info("Got info about artifact with ID {}: {}", artifactId, response.toString());
        assertThat(response.getString("type"), is("INVALID"));
        assertThat(response.getString("config"), is("invalid"));
    }

    @Test
    void createArtifactSpecificVersion() {
        ByteArrayInputStream artifactData = new ByteArrayInputStream("{\"type\":\"record\",\"name\":\"myrecord1\",\"fields\":[{\"name\":\"foo\",\"type\":\"string\"}]}".getBytes(StandardCharsets.UTF_8));
        String artifactId = "createArtifactSpecificVersionId";
        ArtifactMetaData metaData = ArtifactUtils.createArtifact(apicurioService, ArtifactType.AVRO, artifactId, artifactData);
        LOGGER.info("Created artifact {} with metadata {}", artifactId, metaData.toString());

        artifactData = new ByteArrayInputStream("{\"type\":\"record\",\"name\":\"myrecord1\",\"fields\":[{\"name\":\"bar\",\"type\":\"string\"}]}".getBytes(StandardCharsets.UTF_8));
        metaData = ArtifactUtils.updateArtifact(apicurioService, ArtifactType.AVRO, artifactId, artifactData);
        LOGGER.info("Artifact with ID {} was updated: {}", artifactId, metaData.toString());

        List<Long> artifactVersions = apicurioService.listArtifactVersions(artifactId);

        LOGGER.info("Available versions of artifact with ID {} are: {}", artifactId, artifactVersions.toString());
        assertThat(artifactVersions, hasItems(1L, 2L));
    }

    @Test
    void testDuplicatedArtifact() {
        ByteArrayInputStream artifactData = new ByteArrayInputStream("{\"type\":\"record\",\"name\":\"myrecord1\",\"fields\":[{\"name\":\"foo\",\"type\":\"string\"}]}".getBytes(StandardCharsets.UTF_8));
        String artifactId = "duplicateArtifactId";
        ArtifactMetaData metaData = ArtifactUtils.createArtifact(apicurioService, ArtifactType.AVRO, artifactId, artifactData);
        LOGGER.info("Created artifact {} with metadata {}", artifactId, metaData.toString());

        artifactData = new ByteArrayInputStream("{\"type\":\"record\",\"name\":\"alreadyExistArtifact\",\"fields\":[{\"name\":\"foo\",\"type\":\"string\"}]}".getBytes(StandardCharsets.UTF_8));

        try {
            ArtifactUtils.createArtifact(apicurioService, ArtifactType.AVRO, artifactId, artifactData);
        } catch (WebApplicationException e) {
            assertThat("{\"message\":\"An artifact with ID 'duplicateArtifactId' already exists.\",\"error_code\":409}", is(e.getResponse().readEntity(String.class)));
        }
    }
    
    @Test
    void testDisableEnableArtifact() {
        String artifactId = "testDisableEnableArtifact";
        String artifactData = "{\"type\":\"record\",\"name\":\"myrecord1\",\"fields\":[{\"name\":\"foo\",\"type\":\"string\"}]}";
        
        // Create the artifact
        ArtifactMetaData metaData = ArtifactUtils.createArtifact(apicurioService, ArtifactType.AVRO, artifactId, IoUtil.toStream(artifactData));
        LOGGER.info("Created artifact {} with metadata {}", artifactId, metaData.toString());

        // Verify
        ArtifactMetaData actualMD = apicurioService.getArtifactMetaData(artifactId);
        assertEquals(metaData.getGlobalId(), actualMD.getGlobalId());

        // Disable the artifact
        UpdateState data = new UpdateState();
        data.setState(ArtifactState.DISABLED);
        apicurioService.updateArtifactState(artifactId, data);
        
        // Verify (expect 404)
        try {
            apicurioService.getArtifactMetaData(artifactId);
            fail("Expected 404");
        } catch (WebApplicationException e) {
            assertEquals(404, e.getResponse().getStatus());
        }
        
        // Re-enable the artifact
        data.setState(ArtifactState.ENABLED);
        apicurioService.updateArtifactState(artifactId, data);
        
        // Verify
        actualMD = apicurioService.getArtifactMetaData(artifactId);
        assertEquals(metaData.getGlobalId(), actualMD.getGlobalId());
    }

    @Test
    void testDisableEnableArtifactVersion() {
        String artifactId = "testDisableEnableArtifactVersion";
        String artifactData = "{\"type\":\"record\",\"name\":\"myrecord1\",\"fields\":[{\"name\":\"foo\",\"type\":\"string\"}]}";
        String artifactDataV2 = "{\"type\":\"record\",\"name\":\"myrecord2\",\"fields\":[{\"name\":\"foo\",\"type\":\"string\"}]}";
        String artifactDataV3 = "{\"type\":\"record\",\"name\":\"myrecord3\",\"fields\":[{\"name\":\"foo\",\"type\":\"string\"}]}";
        
        // Create the artifact
        ArtifactMetaData v1MD = ArtifactUtils.createArtifact(apicurioService, ArtifactType.AVRO, artifactId, IoUtil.toStream(artifactData));
        LOGGER.info("Created artifact {} with metadata {}", artifactId, v1MD.toString());

        // Update the artifact (v2)
        ArtifactMetaData v2MD = ArtifactUtils.updateArtifact(apicurioService, ArtifactType.AVRO, artifactId, IoUtil.toStream(artifactDataV2));
        
        // Update the artifact (v3)
        ArtifactMetaData v3MD = ArtifactUtils.updateArtifact(apicurioService, ArtifactType.AVRO, artifactId, IoUtil.toStream(artifactDataV3));
        
        // Disable v3
        UpdateState data = new UpdateState();
        data.setState(ArtifactState.DISABLED);
        apicurioService.updateArtifactVersionState(v3MD.getVersion(), artifactId, data);

        // Verify artifact
        ArtifactMetaData actualMD = apicurioService.getArtifactMetaData(artifactId);
        assertEquals(ArtifactState.ENABLED, actualMD.getState());
        assertEquals(2, actualMD.getVersion()); // version 2 is active (3 is disabled)
        
        // Verify v1
        VersionMetaData actualVMD = apicurioService.getArtifactVersionMetaData(v1MD.getVersion(), artifactId);
        assertEquals(ArtifactState.ENABLED, actualVMD.getState());
        // Verify v2
        actualVMD = apicurioService.getArtifactVersionMetaData(v2MD.getVersion(), artifactId);
        assertEquals(ArtifactState.ENABLED, actualVMD.getState());
        // Verify v3
        actualVMD = apicurioService.getArtifactVersionMetaData(v3MD.getVersion(), artifactId);
        assertEquals(ArtifactState.DISABLED, actualVMD.getState());
        
        // Re-enable v3
        data.setState(ArtifactState.ENABLED);
        apicurioService.updateArtifactVersionState(v3MD.getVersion(), artifactId, data);

        // Verify artifact (now v3)
        actualMD = apicurioService.getArtifactMetaData(artifactId);
        assertEquals(ArtifactState.ENABLED, actualMD.getState());
        assertEquals(3, actualMD.getVersion()); // version 2 is active (3 is disabled)

        // Verify v1
        actualVMD = apicurioService.getArtifactVersionMetaData(v1MD.getVersion(), artifactId);
        assertEquals(ArtifactState.ENABLED, actualVMD.getState());
        // Verify v2
        actualVMD = apicurioService.getArtifactVersionMetaData(v2MD.getVersion(), artifactId);
        assertEquals(ArtifactState.ENABLED, actualVMD.getState());
        // Verify v3
        actualVMD = apicurioService.getArtifactVersionMetaData(v3MD.getVersion(), artifactId);
        assertEquals(ArtifactState.ENABLED, actualVMD.getState());
    }
    
    @Test
    void testDeprecateArtifact() {
        String artifactId = "testDeprecateArtifact";
        String artifactData = "{\"type\":\"record\",\"name\":\"myrecord1\",\"fields\":[{\"name\":\"foo\",\"type\":\"string\"}]}";
        
        // Create the artifact
        ArtifactMetaData metaData = ArtifactUtils.createArtifact(apicurioService, ArtifactType.AVRO, artifactId, IoUtil.toStream(artifactData));
        LOGGER.info("Created artifact {} with metadata {}", artifactId, metaData.toString());

        // Verify
        ArtifactMetaData actualMD = apicurioService.getArtifactMetaData(artifactId);
        assertEquals(metaData.getGlobalId(), actualMD.getGlobalId());
        assertEquals(ArtifactState.ENABLED, actualMD.getState());

        // Deprecate the artifact
        UpdateState data = new UpdateState();
        data.setState(ArtifactState.DEPRECATED);
        apicurioService.updateArtifactState(artifactId, data);
        
        // Verify (expect 404)
        actualMD = apicurioService.getArtifactMetaData(artifactId);
        assertEquals(metaData.getGlobalId(), actualMD.getGlobalId());
        assertEquals(ArtifactState.DEPRECATED, actualMD.getState());
    }
    
    @Test
    void testDeprecateArtifactVersion() {
        String artifactId = "testDeprecateArtifactVersion";
        String artifactData = "{\"type\":\"record\",\"name\":\"myrecord1\",\"fields\":[{\"name\":\"foo\",\"type\":\"string\"}]}";
        String artifactDataV2 = "{\"type\":\"record\",\"name\":\"myrecord2\",\"fields\":[{\"name\":\"foo\",\"type\":\"string\"}]}";
        String artifactDataV3 = "{\"type\":\"record\",\"name\":\"myrecord3\",\"fields\":[{\"name\":\"foo\",\"type\":\"string\"}]}";
        
        // Create the artifact
        ArtifactMetaData v1MD = ArtifactUtils.createArtifact(apicurioService, ArtifactType.AVRO, artifactId, IoUtil.toStream(artifactData));
        LOGGER.info("Created artifact {} with metadata {}", artifactId, v1MD.toString());

        // Update the artifact (v2)
        ArtifactMetaData v2MD = ArtifactUtils.updateArtifact(apicurioService, ArtifactType.AVRO, artifactId, IoUtil.toStream(artifactDataV2));
        
        // Update the artifact (v3)
        ArtifactMetaData v3MD = ArtifactUtils.updateArtifact(apicurioService, ArtifactType.AVRO, artifactId, IoUtil.toStream(artifactDataV3));
        
        // Deprecate v2
        UpdateState data = new UpdateState();
        data.setState(ArtifactState.DEPRECATED);
        apicurioService.updateArtifactVersionState(v2MD.getVersion(), artifactId, data);

        // Verify artifact
        ArtifactMetaData actualMD = apicurioService.getArtifactMetaData(artifactId);
        assertEquals(ArtifactState.ENABLED, actualMD.getState());
        
        // Verify v1
        VersionMetaData actualVMD = apicurioService.getArtifactVersionMetaData(v1MD.getVersion(), artifactId);
        assertEquals(ArtifactState.ENABLED, actualVMD.getState());
        // Verify v2
        actualVMD = apicurioService.getArtifactVersionMetaData(v2MD.getVersion(), artifactId);
        assertEquals(ArtifactState.DEPRECATED, actualVMD.getState());
        // Verify v3
        actualVMD = apicurioService.getArtifactVersionMetaData(v3MD.getVersion(), artifactId);
        assertEquals(ArtifactState.ENABLED, actualVMD.getState());
    }

    @Test
    void deleteNonexistingSchema() {
        assertThrows(WebApplicationException.class, () -> apicurioService.deleteArtifact("non-existing"));
    }
    

    @AfterEach
    void deleteRules() {
        apicurioService.deleteAllGlobalRules();
    }
}

