/*******************************************************************************
 * Copyright (c) 2019, 2020 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package com.ibm.ws.security.acme;

import java.security.cert.X509Certificate;
import java.util.List;

/**
 * An interface to define the methods that an ACME 2.0 OSGi service will
 * provide.
 */
public interface AcmeProvider {

	/**
	 * Check whether a new certificate needs to be retrieved from the
	 * certificate authority, and if so retrieve the certificate.
	 * 
	 * @return The certificate chain retrieved from the CA, if one was
	 *         retrieved. Returns null otherwise.
	 * @throws AcmeCaException
	 *             If there was an error checking or retrieving the certificate.
	 */
	public List<X509Certificate> checkAndRetrieveCertificate() throws AcmeCaException;

	/**
	 * Get the HTTP-01 challenge authorization for the specified challenge
	 * token. Both the challenge token and the challenge authorization are
	 * generated by the ACME CA server.
	 * 
	 * @param token
	 *            The HTTP-01 challenge token to get the authorization for.
	 * @return The HTTP-01 challenge authorization.
	 * @throws AcmeCaException
	 *             If there was an error retrieving the HTTP-1 challenge
	 *             authorization.
	 */
	public String getHttp01Authorization(String token) throws AcmeCaException;

	/**
	 * Revoke a certificate using an existing account on the ACME server. If the
	 * account key pair cannot be found, we will fail.
	 * 
	 * @param certificate
	 *            The certificate to revoke.
	 * @throws AcmeCaException
	 *             If there was an error revoking the certificate.
	 */
	public void revoke(X509Certificate certificate) throws AcmeCaException;

}
