#!/bin/sh
#
# Script to unpack, build, and bundle the built-in HQ database
#
#
VERSION=8.1.2

# Shouldn't need to modify anything below this line

cd `dirname $0`
BUILD_DIR=`pwd`

PG_INST=${BUILD_DIR}/pgsql
PG_PACK=${BUILD_DIR}/pgsql.tar.gz
PG_DIST=${BUILD_DIR}/postgresql-${VERSION}.tar.gz
PG_DIR=${BUILD_DIR}/postgresql-${VERSION}

PG_OPTIONS="--prefix=${PG_INST} --without-zlib --without-readline"

echo "Using options: ${PG_OPTIONS}"

if [ ! -f "${PG_DIST}" ] ; then
  echo "PostgreSQL bundle at ${PG_DIST} not found."
  exit 1
fi 

if [ -d "${PG_DIR}" ] ; then
  echo "Existing source tree exists at ${PG_DIR}"
  exit 1
fi

if [ -d "${PG_INST}" ] ; then
  echo "Existing HQ database exists at ${PG_INST}"
  exit 1
fi

# Calculate the release directory
PLATFORM=`uname -s`
ARCH=`uname -m`
case "$PLATFORM" in
  Linux )
    if [ "${ARCH}" = "x86_64" ]; then
      RELEASE_DIR=${BUILD_DIR}/amd64-linux
    else
      RELEASE_DIR=${BUILD_DIR}/x86-linux     
    fi
    ;;
  SunOS )
    RELEASE_DIR=${BUILD_DIR}/sparc-solaris
    ;;
  Darwin )
    RELEASE_DIR=${BUILD_DIR}/apple-darwin
    ;;
  HP-UX )
    RELEASE_DIR=${BUILD_DIR}/hpux-11
    ;;
  * )
    echo "Unsupported platform: ${PLATFORM}"
    exit 1
    ;;
esac

# Begin build

gunzip -c ${PG_DIST} | tar -vxf -

cd ${PG_DIR}

./configure ${PG_OPTIONS}

if [ $? -ne 0 ] ; then
  exit 1
fi

make

if [ $? -ne 0 ] ; then
  exit 1
fi

make install

if [ $? -ne 0 ] ; then
  exit 1
fi

# Configure

cd ${PG_INST}

LANG=C ./bin/initdb -D data

if [ $? -ne 0 ] ; then
  exit 1
fi

./bin/pg_ctl start -o -i -D data

if [ $? -ne 0 ] ; then
  exit 1
fi

# Wait for postmaster to start
echo "Waiting for postmaster to start"
sleep 10

./bin/createdb hqdb

if [ $? -ne 0 ] ; then
  exit 1
fi

./bin/createuser -s hqadmin

if [ $? -ne 0 ] ; then
  exit 1
fi

./bin/pg_ctl stop -o -i -D data

# Remove extras
rm -rf doc include man
rm -f data/*.conf data/*.opts

# Create dir.links

perl ${BUILD_DIR}/dir-links.pl 

# Pack

tar cvf - . | gzip > ${PG_PACK}

echo "Copying to ${RELEASE_DIR}"

mv ${PG_PACK} ${RELEASE_DIR}

rm -rf ${PG_INST} ${PG_DIR}

exit 0
