/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.spring.httpclient.resource;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.hc.client5.http.cache.CacheResponseStatus;
import org.apache.hc.client5.http.cache.HttpCacheContext;
import org.apache.hc.client5.http.classic.HttpClient;
import org.apache.hc.client5.http.classic.methods.HttpGet;
import org.apache.hc.client5.http.classic.methods.HttpUriRequest;
import org.apache.hc.client5.http.utils.DateUtils;
import org.apache.hc.core5.http.ClassicHttpResponse;
import org.apache.hc.core5.http.Header;
import org.apache.hc.core5.http.HttpHeaders;
import org.apache.hc.core5.http.HttpResponse;
import org.apache.hc.core5.http.HttpStatus;
import org.apache.hc.core5.http.io.entity.EntityUtils;
import org.slf4j.Logger;

import org.springframework.beans.factory.BeanNameAware;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.core.io.Resource;

import net.shibboleth.shared.annotation.ParameterName;
import net.shibboleth.shared.annotation.constraint.NotEmpty;
import net.shibboleth.shared.component.AbstractIdentifiedInitializableComponent;
import net.shibboleth.shared.component.ComponentInitializationException;
import net.shibboleth.shared.httpclient.HttpClientContextHandler;
import net.shibboleth.shared.logic.Constraint;
import net.shibboleth.shared.primitive.LoggerFactory;
import net.shibboleth.shared.primitive.StringSupport;

/**
 * Resource for looking up HTTP URLs. Allows injection and therefore configuration of an Apache {@link HttpClient}. Code
 * based on OpenSAML <code>HTTPMetadataResolver</code> and {@link org.springframework.core.io.UrlResource}.
 */
public class HTTPResource extends AbstractIdentifiedInitializableComponent implements Resource, BeanNameAware,
        InitializingBean, net.shibboleth.shared.resource.Resource {

    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(HTTPResource.class);

    /** HTTP Client used to pull the resource. */
    @Nonnull private final HttpClient httpClient;
    
    /** URL to the Resource. */
    @Nonnull private final URL resourceURL;
    
    /** Optional handler to pre- and post-process context. */
    @Nullable private HttpClientContextHandler httpClientContextHandler;

    /**
     * Constructor.
     * 
     * @param client the client we use to connect with.
     * @param url URL to the remote data
     * @throws IOException if the URL was badly formed
     */
    public HTTPResource(@Nonnull @ParameterName(name="client") final HttpClient client,
            @Nonnull @NotEmpty @ParameterName(name="url") final String url) throws IOException {
        httpClient = Constraint.isNotNull(client, "HttpClient cannot be null");
        final String trimmedAddress =
                Constraint.isNotNull(StringSupport.trimOrNull(url), "Provided URL cannot be null or empty");
        resourceURL = new URL(trimmedAddress);

    }

    /**
     * Constructor.
     * 
     * @param client the client we use to connect with.
     * @param url URL to the remote data
     * @throws IOException if the URL was badly formed
     */
    public HTTPResource(@Nonnull @ParameterName(name="") final HttpClient client,
            @Nonnull @ParameterName(name="url") final URL url) throws IOException {
        httpClient = Constraint.isNotNull(client, "HttpClient cannot be null");
        resourceURL = Constraint.isNotNull(url, "Provided URL cannot be null or empty");

    }
    
    /**
     * Set a handler to manipulate the {@link org.apache.hc.client5.http.protocol.HttpClientContext}.
     * 
     * @param handler the handler to install
     * 
     * @since 5.4.0
     */
    public void setHttpClientContextHandler(@Nonnull final HttpClientContextHandler handler) {
        checkSetterPreconditions();
        
        httpClientContextHandler = handler;
    }

    /**
     * Build the {@link HttpCacheContext} instance which will be used to invoke the {@link HttpClient} request.
     * 
     * @return a new instance of {@link HttpCacheContext}
     */
    @Nonnull protected HttpCacheContext buildHttpClientContext() {
        final HttpCacheContext result = HttpCacheContext.create();
        assert result != null;
        return result;
    }

    /**
     * Print out to the log whether we hit the apache cache or not.
     * http://hc.apache.org/httpcomponents-client-ga/tutorial/html/caching.html
     * @param context the context of the request
     */
    protected void reportCachingStatus(@Nonnull final HttpCacheContext context) {
        final CacheResponseStatus responseStatus = context.getCacheResponseStatus();
        if (null == responseStatus) {
            log.debug("Non caching client provided");
            return;
        }
        switch (responseStatus) {
            case CACHE_HIT:
                log.debug("A response was generated from the cache with no requests sent upstream");
                break;
            case CACHE_MODULE_RESPONSE:
                log.debug("The response was generated directly by the caching module");
                break;
            case CACHE_MISS:
                log.debug("The response came from an upstream server");
                break;
            case VALIDATED:
                log.debug("The response was generated from the cache "
                        + "after validating the entry with the origin server");
                break;
            default:
                log.info("Unknown status {}", responseStatus.toString());
                break;
        }
    }

    /** {@inheritDoc} */
    @Override @Nonnull public InputStream getInputStream() throws IOException {
        final HttpGet httpGet = new HttpGet(resourceURL.toExternalForm());
        final HttpCacheContext context = buildHttpClientContext();

        final HttpClientContextHandler contextHandler = httpClientContextHandler;
        
        if (contextHandler != null) {
            log.debug("Invoking HttpClientContextHandler prior to execution");
            contextHandler.invokeBefore(context, httpGet);
        }
        
        log.debug("Attempting to get data from remote resource '{}'", resourceURL);
        final ClassicHttpResponse response = httpClient.executeOpen(null, httpGet, context);
        
        if (contextHandler != null) {
            log.debug("Invoking HttpClientContextHandler after execution");
            try {
                contextHandler.invokeAfter(context, httpGet);
            } catch (final Throwable e) {
                closeResponse(response);
                throw e;
            }
        }
        
        reportCachingStatus(context);
        final int httpStatusCode = response.getCode();

        if (httpStatusCode != HttpStatus.SC_OK) {
            final String errMsg =
                    "Non-ok status code " + httpStatusCode + " returned from remote resource " + resourceURL;
            log.error(errMsg);
            closeResponse(response);
            throw new IOException(errMsg);
        }

        return new ConnectionClosingInputStream(response);
    }

    /** {@inheritDoc} */
    @Override public void afterPropertiesSet() throws Exception {
        initialize();
    }

    /** {@inheritDoc} */
    @Override public void setBeanName(@Nonnull final String name) {
        setId(name);
    }

    /** {@inheritDoc} */
    @Override public boolean exists() {

        log.debug("Attempting to fetch remote resource as '{}'", resourceURL);
        final HttpResponse response;
        try {
            response = getResourceHeaders();
        } catch (final IOException e) {
            return false;
        }
        final int httpStatusCode = response.getCode();

        return httpStatusCode == HttpStatus.SC_OK;
    }

    /** {@inheritDoc} */
    @Override public boolean isFile() {
        return false;
    }

    /** {@inheritDoc} */
    @Override public boolean isReadable() {
        return true;
    }

    /** {@inheritDoc} */
    @Override public boolean isOpen() {
        return false;
    }

    /** {@inheritDoc} */
    @Override @Nonnull public URL getURL() throws IOException {
        return resourceURL;
    }

    /** {@inheritDoc} */
    @Override @Nonnull public URI getURI() throws IOException {
        try {
            final URI result = resourceURL.toURI();
            assert result != null;
            return result;
        } catch (final URISyntaxException ex) {
            throw new IOException("Invalid URI [" + resourceURL + "]", ex);
        }
    }

    /** {@inheritDoc} Based on {@link org.springframework.core.io.UrlResource}. */
    @Override @Nonnull public File getFile() throws IOException {
        throw new FileNotFoundException("HTTPResource cannot be resolved to absolute file path "
                + "because it does not reside in the file system: " + resourceURL);
    }

    /**
     * Attempts to fetch only the headers for a given resource. If HEAD requests are unsupported then a more costly GET
     * request is performed.
     * 
     * <b>NOTE</b> This method returns a <b>closed</b> {@link ClassicHttpResponse}.  See JSSH-25.
     * It turns out that for the cases we need it this is OK and all three of them have regression
     * tests.  If they fail, this will be revisisted.
     * 
     * @return the response from the request
     * 
     * @throws IOException thrown if there is a problem contacting the resource
     */
    @Nonnull protected HttpResponse getResourceHeaders() throws IOException {
        final HttpUriRequest httpRequest = new HttpGet(resourceURL.toExternalForm());

        ClassicHttpResponse httpResponse = null;
        try {
            final HttpCacheContext context = buildHttpClientContext();
            final HttpClientContextHandler contextHandler = httpClientContextHandler;
            
            if (contextHandler != null) {
                log.debug("Invoking HttpClientContextHandler prior to execution");
                contextHandler.invokeBefore(context, httpRequest);
            }
            
            httpResponse = httpClient.executeOpen(null, httpRequest, context);
            
            if (contextHandler != null) {
                log.debug("Invoking HttpClientContextHandler after execution");
                contextHandler.invokeAfter(context, httpRequest);
            }
            
            reportCachingStatus(context);
            EntityUtils.consume(httpResponse.getEntity());
            return httpResponse;
        } catch (final IOException e) {
            throw new IOException("Error contacting remote resource " + resourceURL.toString(), e);
        } finally {
            closeResponse(httpResponse);
        }
    }

    /**
     * Send a Head to the client and interrogate the response for a particular response header.
     * 
     * @param what the repsonse header to look at
     * @return the value of that response, or null if things failed
     * @throws IOException from lower levels.
     */
    @Nullable protected String getResponseHeader(final String what) throws IOException {

        log.debug("Attempting to fetch remote resource as '{}'", resourceURL);
        final HttpResponse response = getResourceHeaders();
        final int httpStatusCode = response.getCode();

        if (httpStatusCode != HttpStatus.SC_OK) {
            final String errMsg =
                    "Non-ok status code " + httpStatusCode + " returned from remote resource " + resourceURL;
            log.error(errMsg);
            throw new IOException(errMsg);
        }

        final Header httpHeader = response.getFirstHeader(what);
        if (httpHeader != null) {
            return httpHeader.getValue();
        }
        return null;
    }

    /** {@inheritDoc} */
    @Override public long contentLength() throws IOException {

        final String response = getResponseHeader(HttpHeaders.CONTENT_LENGTH);
        if (null != response) {
            return Long.parseLong(response);
        }
        final String errMsg = "Response from remote resource " + resourceURL.toString() + 
                " did not contain a Content-Length header";
        log.error(errMsg);
        throw new IOException(errMsg);
    }

    /** {@inheritDoc} */
    @Override public long lastModified() throws IOException {
        final String response = getResponseHeader(HttpHeaders.LAST_MODIFIED);
        if (null != response) {
            return DateUtils.parseStandardDate(response).toEpochMilli();
        }
        final String errMsg = "Response from remote resource " + resourceURL.toString() + 
                " did not contain a Last-Modified header";
        log.error(errMsg);
        throw new IOException(errMsg);
    }

    /** {@inheritDoc} Based on {@link org.springframework.core.io.UrlResource}. */
    @Override @Nonnull public HTTPResource createRelative(@Nonnull final String relativePath) throws IOException {
        return createRelative(relativePath, null);
    }
        
    /** Create a resource relative to us, but use the supplied handler.
     *
     * @param relativePath The path to create
     * @param handler (optional) the handler
     * @return the resource
     * @throws IOException if we cannot set things up correctly
     * @since 9.0.0
     */
     @Nonnull public HTTPResource createRelative(@Nonnull final String relativePath,
             @Nullable final HttpClientContextHandler handler) throws IOException {

        final String path;
        if (relativePath.startsWith("/")) {
            path = relativePath.substring(1);
        } else {
            path = relativePath;
        }
        final HTTPResource result = new HTTPResource(httpClient, new URL(resourceURL, path));
        if (handler!=null) {
            result.setHttpClientContextHandler(handler);
        }
        try {
            result.setId(getId()+"_"+relativePath);
            result.initialize();
        } catch (final ComponentInitializationException e) {
            throw new IOException(e);
        }
        return result;
    }

    /** {@inheritDoc} */
    @Override @Nonnull public net.shibboleth.shared.resource.Resource createRelativeResource(
            @Nonnull final String relativePath) throws IOException {

        return createRelative(relativePath);
    }

    /**
     * {@inheritDoc} This implementation returns the name of the file that this URL refers to.
     * 
     * @see java.net.URL#getFile()
     * @see java.io.File#getName()
     */
    @Override @Nullable public String getFilename() {
        return new File(resourceURL.getFile()).getName();
    }

    /** {@inheritDoc} */
    @Override @Nonnull public String getDescription() {
        final StringBuilder builder = new StringBuilder("HTTPResource [").append(resourceURL.toString()).append(']');
        final String result = builder.toString();
        assert result != null;
        return result;

    }
    
    /**
     * Close the HTTP response.
     * 
     * @param response the HTTP response
     */
    protected void closeResponse(@Nullable final ClassicHttpResponse response) {
        try {
            if (response != null) {
                response.close();
            }
        } catch (final IOException e) {
            log.error("Error closing HTTP response from '{}'", resourceURL.toExternalForm(), e);
        }
    }

}