/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.spring.httpclient.resource;

import java.io.IOException;

import net.shibboleth.shared.component.ComponentInitializationException;
import net.shibboleth.shared.httpclient.HttpClientBuilder;
import net.shibboleth.shared.httpclient.HttpClientContextHandler;
import net.shibboleth.shared.spring.custom.SchemaTypeAwareXMLBeanDefinitionReader;
import net.shibboleth.shared.spring.resource.ConditionalResource;
import net.shibboleth.shared.spring.resource.ConditionalResourceResolver;

import org.apache.hc.client5.http.classic.HttpClient;
import org.apache.hc.client5.http.protocol.HttpClientContext;
import org.apache.hc.core5.http.ClassicHttpRequest;
import org.springframework.beans.factory.xml.XmlBeanDefinitionReader;
import org.springframework.context.support.GenericApplicationContext;
import org.springframework.core.io.ClassPathResource;
import org.testng.Assert;
import org.testng.annotations.BeforeClass;
import org.testng.annotations.Test;

/**
 * Test for {@link ConditionalResource}.
 */
@SuppressWarnings("javadoc")
public class ConditionalResourceTest {

    private final String documentPath = "/net/shibboleth/shared/spring/httpclient/resource/document.xml";

    private final String existsURL = RepositorySupport.buildHTTPResourceURL("java-shib-shared", "shib-networking-spring/src/test/resources/net/shibboleth/shared/spring/httpclient/resource/document.xml",false);

    private final String nonExistsURL = RepositorySupport.buildHTTPResourceURL("java-shib-shared", "trunk/src/test/resources/data/document.xml",false);

    private HttpClient client;

    @BeforeClass public void setupClient() throws Exception {
        client = (new HttpClientBuilder()).buildClient();
    }

    @Test public void existsTest() throws IOException, ComponentInitializationException {
        final HTTPResource existsHTTPResource = new HTTPResource(client, existsURL);
        final HTTPResource notExistsHTTPResource = new HTTPResource(client, nonExistsURL);
        
        final ConditionalResource existsResource = new ConditionalResource(existsHTTPResource);
        final ConditionalResource notExistsResource = new ConditionalResource(notExistsHTTPResource);
        
        existsResource.setId("test");
        existsResource.initialize();

        notExistsResource.setId("test");
        notExistsResource.initialize();

        Assert.assertTrue(existsHTTPResource.exists());
        Assert.assertFalse(notExistsHTTPResource.exists());

        Assert.assertTrue(existsResource.exists());
        Assert.assertTrue(notExistsResource.exists());
    }
    
    
    @Test public void contextHandlerFailBeforeTest() throws IOException, ComponentInitializationException {
        final HTTPResource existsHTTPResource = new HTTPResource(client, existsURL);
        existsHTTPResource.setHttpClientContextHandler(new HttpClientContextHandler() {
            public void invokeBefore(HttpClientContext context, ClassicHttpRequest request) throws IOException {
                throw new IOException("Fail");
            }
            public void invokeAfter(HttpClientContext context, ClassicHttpRequest request) throws IOException {
            }
        });
        
        final ConditionalResource existsResource = new ConditionalResource(existsHTTPResource);
        existsResource.setId("test");
        existsResource.initialize();

        Assert.assertTrue(existsResource.exists());
    }
    
    @Test public void contextHandlerFailAfterTest() throws IOException, ComponentInitializationException {
        final HTTPResource existsHTTPResource = new HTTPResource(client, existsURL);
        existsHTTPResource.setHttpClientContextHandler(new HttpClientContextHandler() {
            public void invokeBefore(HttpClientContext context, ClassicHttpRequest request) throws IOException {
            }
            public void invokeAfter(HttpClientContext context, ClassicHttpRequest request) throws IOException {
                throw new IOException("Fail");
            }
        });

        final ConditionalResource existsResource = new ConditionalResource(existsHTTPResource);
        existsResource.setId("test");
        existsResource.initialize();
        
        Assert.assertTrue(existsResource.exists());
    }

    @Test public void testCompare() throws IOException, ComponentInitializationException {

        final HTTPResource existsHTTPResource = new HTTPResource(client, existsURL);
        
        final ConditionalResource existsResource = new ConditionalResource(existsHTTPResource);
        existsResource.setId("test");
        existsResource.initialize();

        Assert.assertTrue(ResourceTestHelper.compare(existsResource, new ClassPathResource(documentPath)));
    }
    
    @Test public void testBeanExists() throws ComponentInitializationException {
        final ClassPathResource existsCPResource =
                new ClassPathResource("net/shibboleth/shared/spring/httpclient/resource/conditional.xml");
        final ConditionalResource existsResource = new ConditionalResource(existsCPResource);
        existsResource.setId("test");
        existsResource.initialize();
        
        final GenericApplicationContext parentContext = new GenericApplicationContext();
        parentContext.refresh(); // THIS IS REQUIRED

        final GenericApplicationContext context = new GenericApplicationContext(parentContext);
        final XmlBeanDefinitionReader beanDefinitionReader = new SchemaTypeAwareXMLBeanDefinitionReader(context);

        beanDefinitionReader.setValidationMode(XmlBeanDefinitionReader.VALIDATION_XSD);
        beanDefinitionReader.loadBeanDefinitions(existsResource);
        context.refresh();
        
        Assert.assertEquals(context.getBean("testBean"), "foo");
    }
    
    @Test public void testBeanMissing() throws ComponentInitializationException {
        final ClassPathResource missingCPResource =
                new ClassPathResource("net/shibboleth/shared/spring/httpclient/resource/missing.xml");
        final ConditionalResource missingResource = new ConditionalResource(missingCPResource);
        missingResource.setId("test");
        missingResource.initialize();
        
        final GenericApplicationContext parentContext = new GenericApplicationContext();
        parentContext.refresh(); // THIS IS REQUIRED

        final GenericApplicationContext context = new GenericApplicationContext(parentContext);
        final XmlBeanDefinitionReader beanDefinitionReader = new SchemaTypeAwareXMLBeanDefinitionReader(context);

        beanDefinitionReader.setValidationMode(XmlBeanDefinitionReader.VALIDATION_XSD);
        beanDefinitionReader.loadBeanDefinitions(missingResource);
        context.refresh();
        
        Assert.assertFalse(context.containsBean("testBean"));
    }
    
    @Test public void testImport() {
        final ClassPathResource resource =
                new ClassPathResource("net/shibboleth/shared/spring/httpclient/resource/conditional-import.xml");
        
        final GenericApplicationContext parentContext = new GenericApplicationContext();
        parentContext.refresh(); // THIS IS REQUIRED

        final GenericApplicationContext context = new GenericApplicationContext(parentContext);
        final XmlBeanDefinitionReader beanDefinitionReader = new SchemaTypeAwareXMLBeanDefinitionReader(context);
        context.addProtocolResolver(new ConditionalResourceResolver());

        beanDefinitionReader.setValidationMode(XmlBeanDefinitionReader.VALIDATION_XSD);
        beanDefinitionReader.loadBeanDefinitions(resource);
        context.refresh();
        
        Assert.assertEquals(context.getBean("testBean"), "foo");
    }
    
}