/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.spring.httpclient.resource;

import java.io.File;
import java.io.IOException;

import net.shibboleth.shared.collection.CollectionSupport;
import net.shibboleth.shared.httpclient.HttpClientBuilder;
import net.shibboleth.shared.spring.util.ApplicationContextBuilder;

import org.apache.hc.client5.http.classic.HttpClient;
import org.springframework.context.support.GenericApplicationContext;
import org.springframework.core.io.ClassPathResource;
import org.springframework.core.io.Resource;
import org.springframework.mock.env.MockPropertySource;
import org.testng.Assert;
import org.testng.annotations.AfterClass;
import org.testng.annotations.BeforeClass;
import org.testng.annotations.Test;

/**
 * Test for FileBackedHTTPResource.
 */
@SuppressWarnings("javadoc")
public class FileBackedHTTPResourceTest {

    private final String documentPath = "/net/shibboleth/shared/spring/httpclient/resource/document.xml";

    private final String existsURL =
            RepositorySupport.buildHTTPResourceURL("java-shib-shared", "shib-networking-spring/src/test/resources/net/shibboleth/shared/spring/httpclient/resource/document.xml",false);

    private final String nonExistsURL = RepositorySupport.buildHTTPResourceURL("java-shib-shared", "trunk/src/test/resources/data/document.xml",false);

    private String existsFile;

    private HttpClient client;

    @BeforeClass public void setupClient() throws Exception {
        client = (new HttpClientBuilder()).buildClient();
        final File file = File.createTempFile("FileBackedHTTPResourceTest1", ".xml");
        existsFile = file.getAbsolutePath();
    }

    @AfterClass public void deleteFile() {
        final File f = new File(existsFile);
        if (f.exists()) {
            f.delete();
        }
    }

    @Test public void existsTest() throws IOException {
        final Resource existsResource = new FileBackedHTTPResource(existsFile, client, existsURL);
        final Resource notExistsResource =
                new FileBackedHTTPResource(existsFile + "ZZZ", client, nonExistsURL);

        Assert.assertTrue(existsResource.exists());
        Assert.assertFalse(notExistsResource.exists());
    }

    @Test public void testCompare() throws IOException {
        Assert.assertTrue(ResourceTestHelper.compare(new FileBackedHTTPResource(existsFile, client, existsURL),
                new ClassPathResource(documentPath)));
        // With that done compare via the backup
        Assert.assertTrue(ResourceTestHelper.compare(new FileBackedHTTPResource(existsFile, client, nonExistsURL),
                new ClassPathResource(documentPath)));
    }

    public GenericApplicationContext getContext(final String location) {

        final MockPropertySource mockEnvVars = new MockPropertySource();
        mockEnvVars.setProperty("file.name", existsFile);
        mockEnvVars.setProperty("the.url", existsURL);

        final ApplicationContextBuilder builder = new ApplicationContextBuilder();
        builder.setUnresolvedServiceConfigurations(CollectionSupport.singletonList(location));
        builder.setPropertySources(CollectionSupport.singletonList(mockEnvVars));

        return builder.build();
    }

    @Test public void testParsingNew() throws IOException {

        try (final GenericApplicationContext context = getContext("net/shibboleth/shared/spring/httpclient/resource/newStyle.xml")) {

            Assert.assertTrue(ResourceTestHelper.compare(context.getBean("namedString", FileBackedHTTPResource.class),
                    new ClassPathResource(documentPath)));
            Assert.assertTrue(ResourceTestHelper.compare(context.getBean("namedURL", FileBackedHTTPResource.class),
                    new ClassPathResource(documentPath)));
            Assert.assertTrue(ResourceTestHelper.compare(context
                    .getBean("numberedString", FileBackedHTTPResource.class),
                    new ClassPathResource(documentPath)));
            Assert.assertTrue(ResourceTestHelper.compare(context.getBean("numberedURL", FileBackedHTTPResource.class),
                    new ClassPathResource(documentPath)));
        }
    }

}
