/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.spring.httpclient.resource;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.time.Instant;
import java.util.Collection;

import org.apache.hc.client5.http.cache.CacheResponseStatus;
import org.apache.hc.client5.http.classic.HttpClient;
import org.apache.hc.client5.http.impl.io.PoolingHttpClientConnectionManager;
import org.apache.hc.client5.http.protocol.HttpClientContext;
import org.apache.hc.core5.http.ClassicHttpRequest;
import org.springframework.beans.factory.xml.XmlBeanDefinitionReader;
import org.springframework.context.ApplicationContext;
import org.springframework.context.support.GenericApplicationContext;
import org.springframework.core.io.ClassPathResource;
import org.testng.Assert;
import org.testng.annotations.BeforeClass;
import org.testng.annotations.Test;

import net.shibboleth.shared.httpclient.HttpClientBuilder;
import net.shibboleth.shared.httpclient.HttpClientContextHandler;
import net.shibboleth.shared.httpclient.InMemoryCachingHttpClientBuilder;
import net.shibboleth.shared.spring.custom.SchemaTypeAwareXMLBeanDefinitionReader;

/**
 * Test for HTTPResource.
 */
@SuppressWarnings("javadoc")
public class HTTPResourceTest {

    private final String documentPath = "/net/shibboleth/shared/spring/httpclient/resource/document.xml";
    
    private final String existsURL =
            "http://git.shibboleth.net/view/?p=java-shib-shared.git;a=blob_plain;f=shib-networking-spring/src/test/resources/net/shibboleth/shared/spring/httpclient/resource/document.xml;h=e8ec7c0d20c7a6b8193e1868398cda0c28df45ed;hb=HEAD";
//    private final String existsURL =
//            RepositorySupport.buildHTTPResourceURL("java-shib-shared", "shib-networking-spring/src/test/resources/net/shibboleth/shared/spring/httpclient/resource/document.xml",false);

    private final String nonExistsURL = RepositorySupport.buildHTTPResourceURL("java-shib-shared", "trunk/src/test/resources/data/document.xml",false);

    private HttpClient client;

    @BeforeClass public void setupClient() throws Exception {
        client = (new HttpClientBuilder()).buildClient();
    }

    @Test public void existsTest() throws IOException {
        final HTTPResource existsResource = new HTTPResource(client, existsURL);
        final HTTPResource notExistsResource = new HTTPResource(client, nonExistsURL);

        Assert.assertTrue(existsResource.exists());
        Assert.assertFalse(notExistsResource.exists());
    }
    
    @Test public void contextHandlerNoopTest() throws IOException {
        final HTTPResource existsResource = new HTTPResource(client, existsURL);
        existsResource.setHttpClientContextHandler(new HttpClientContextHandler() {
            public void invokeBefore(HttpClientContext context, ClassicHttpRequest request) throws IOException {
            }
            public void invokeAfter(HttpClientContext context, ClassicHttpRequest request) throws IOException {
            }
        });

        Assert.assertTrue(existsResource.exists());
    }
    
    @Test public void contextHandlerFailBeforeTest() throws IOException {
        final HTTPResource existsResource = new HTTPResource(client, existsURL);
        existsResource.setHttpClientContextHandler(new HttpClientContextHandler() {
            public void invokeBefore(HttpClientContext context, ClassicHttpRequest request) throws IOException {
                throw new IOException("Fail");
            }
            public void invokeAfter(HttpClientContext context, ClassicHttpRequest request) throws IOException {
            }
        });

        Assert.assertFalse(existsResource.exists());
    }
    
    @Test public void contextHandlerFailAfterTest() throws IOException {
        final HTTPResource existsResource = new HTTPResource(client, existsURL);
        existsResource.setHttpClientContextHandler(new HttpClientContextHandler() {
            public void invokeBefore(HttpClientContext context, ClassicHttpRequest request) throws IOException {
            }
            public void invokeAfter(HttpClientContext context, ClassicHttpRequest request) throws IOException {
                throw new IOException("Fail");
            }
        });

        Assert.assertFalse(existsResource.exists());
    }

    @Test public void testCompare() throws IOException {

        Assert.assertTrue(ResourceTestHelper.compare(new HTTPResource(client, existsURL), new ClassPathResource(documentPath)));
    }

    @Test public void testRelated() throws IOException {

        // Chose a file unlikely to change. Do not use the svn thing because the date will not be there

        final HTTPResource parent =
                new HTTPResource(client, "http://test.shibboleth.net/downloads/identity-provider/archive/2.0.0/");
        final HTTPResource child = parent.createRelative("shibboleth-idp-2.0.0-bin.zip");

        final long when = child.lastModified();
        final long size = child.contentLength();
        final String whenAsString = Instant.ofEpochMilli(when).toString();

        Assert.assertEquals(when, 1588006169000L, "Expected date of " + whenAsString + " did not match)");
        Assert.assertEquals(size, 20784226L, "Size mismatch");
    }

    @Test public void testCachedNoCache() throws IOException, InterruptedException {

        final TestHTTPResource what = new TestHTTPResource(client, existsURL);
        Assert.assertTrue(what.exists());
        Assert.assertNull(what.getLastCacheResponseStatus());
        Assert.assertTrue(ResourceTestHelper.compare(what, new ClassPathResource(documentPath)));
        Assert.assertNull(what.getLastCacheResponseStatus());
    }

    @Test public void testCachedCache() throws Exception {

        final InMemoryCachingHttpClientBuilder builder = new InMemoryCachingHttpClientBuilder();
        builder.setMaxCacheEntries(3);
        final TestHTTPResource what = new TestHTTPResource(builder.buildClient(), existsURL);
        Assert.assertTrue(what.exists());
        Assert.assertNotNull(what.getLastCacheResponseStatus());
        Assert.assertTrue(ResourceTestHelper.compare(what, new ClassPathResource(documentPath)));

        Assert.assertEquals(what.getLastCacheResponseStatus(), CacheResponseStatus.CACHE_HIT);
    }

    private GenericApplicationContext getContext(final String fileName, final File theDir) {
        final GenericApplicationContext parentContext = new GenericApplicationContext();
        parentContext.refresh(); // THIS IS REQUIRED
        if (theDir != null) {
            // Spring 5 blows up on a null registration.
            parentContext.getBeanFactory().registerSingleton("theDir", theDir);
        }
        
        final GenericApplicationContext context = new GenericApplicationContext(parentContext);
        final XmlBeanDefinitionReader beanDefinitionReader = new SchemaTypeAwareXMLBeanDefinitionReader(context);

        beanDefinitionReader.setValidationMode(XmlBeanDefinitionReader.VALIDATION_XSD);
        beanDefinitionReader.loadBeanDefinitions(fileName);
        context.refresh();
        return context;
    }

    @Test public void springLoadMemCache() throws IOException {

        final GenericApplicationContext context =
                getContext("classpath:/net/shibboleth/shared/spring/httpclient/resource/MemBackedHTTPBean.xml", null);
        try {
            final Collection<TestHTTPResource> beans = context.getBeansOfType(TestHTTPResource.class).values();
            Assert.assertEquals(beans.size(), 1);

            final TestHTTPResource what = beans.iterator().next();

            Assert.assertTrue(what.exists());
            Assert.assertNotNull(what.getLastCacheResponseStatus());
            Assert.assertTrue(ResourceTestHelper.compare(what, new ClassPathResource(documentPath)));

            Assert.assertEquals(what.getLastCacheResponseStatus(), CacheResponseStatus.CACHE_HIT);
        } finally {
            final ApplicationContext parent = context.getParent();
            if (parent instanceof GenericApplicationContext) {
                ((GenericApplicationContext) parent).close();
            }
            context.close();
        }
    }

    private void emptyDir(final File dir) {
        for (final File f : dir.listFiles()) {
            if (f.isDirectory()) {
                emptyDir(f);
            }
            Assert.assertTrue(f.delete());
        }
        Assert.assertTrue(dir.delete());
    }

    @Test public void springLoadFileCache() throws IOException {
        File theDir = null;
        GenericApplicationContext context = null;
        try {
            final Path p = Files.createTempDirectory("HTTPResourceTest");
            theDir = p.toFile();
            context = getContext("classpath:/net/shibboleth/shared/spring/httpclient/resource/MemBackedHTTPBean.xml", null);
            final Collection<TestHTTPResource> beans = context.getBeansOfType(TestHTTPResource.class).values();
            Assert.assertEquals(beans.size(), 1);

            final TestHTTPResource what = beans.iterator().next();

            Assert.assertTrue(what.exists());
            Assert.assertNotNull(what.getLastCacheResponseStatus());
            Assert.assertTrue(ResourceTestHelper.compare(what, new ClassPathResource(documentPath)));

            Assert.assertEquals(what.getLastCacheResponseStatus(), CacheResponseStatus.CACHE_HIT);
        } finally {
            if (null != theDir) {
                emptyDir(theDir);
            }
            if (null != context) {
                final ApplicationContext parent = context.getParent();
                if (parent instanceof GenericApplicationContext) {
                    ((GenericApplicationContext) parent).close();
                }
            }
        }
    }

    @Test(timeOut = 5000) public void testCloseResponse() {
        // See IDP-969. This test will timeout if the response is not closed.
        try (final PoolingHttpClientConnectionManager connMgr = new PoolingHttpClientConnectionManager()) {
            final HTTPResource notExistsResource = new HTTPResource(client, nonExistsURL);
            int count = 0;
            while (count <= connMgr.getDefaultMaxPerRoute()) {
                count++;
                try {
                    notExistsResource.getInputStream();
                } catch (final IOException e) {
                    // expected because resource does not exist
                }
            }
        } catch (final IOException e) {
            Assert.fail("Bad URL", e);
        }
    }
}
