/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.spring.servlet.impl;

import org.springframework.mock.web.MockHttpServletRequest;
import org.springframework.mock.web.MockHttpServletResponse;
import org.testng.Assert;
import org.testng.annotations.AfterMethod;
import org.testng.annotations.BeforeMethod;
import org.testng.annotations.Test;

import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import net.shibboleth.shared.servlet.impl.HttpServletRequestResponseContext;

/**
 * Tests for {@link HttpServletRequestResponseContext}.
 */
@SuppressWarnings("javadoc")
public class HttpServletRequestResponseContextTest {
    
    private HttpServletRequest request;
    private HttpServletResponse response;
    
    @BeforeMethod
    public void setUp() {
        final MockHttpServletRequest mockRequest = new MockHttpServletRequest();
        mockRequest.setMethod("GET");
        mockRequest.setRequestURI("/foo");
        mockRequest.addHeader("MyRequestHeader", "MyRequestHeaderValue");
        mockRequest.addParameter("MyParam", "MyParamValue");
        request = mockRequest;
        
        final MockHttpServletResponse mockResponse = new MockHttpServletResponse();
        mockResponse.setHeader("MyResponseHeader", "MyResponseHeaderValue");
        response = mockResponse;
    }
    
    @AfterMethod
    public void tearDown()  {
        HttpServletRequestResponseContext.clearCurrent();
    }
    
    @Test
    public void testLoadAndClear() {
       Assert.assertNull(HttpServletRequestResponseContext.getRequest()); 
       Assert.assertNull(HttpServletRequestResponseContext.getResponse()); 
       
       HttpServletRequestResponseContext.loadCurrent(request, response);
       
       Assert.assertNotNull(HttpServletRequestResponseContext.getRequest()); 
       Assert.assertNotNull(HttpServletRequestResponseContext.getResponse()); 
       
       final HttpServletRequest request = HttpServletRequestResponseContext.getRequest();
       assert(request != null);
       Assert.assertEquals(request.getMethod(), "GET");
       Assert.assertEquals(request.getRequestURI(), "/foo");
       Assert.assertEquals(request.getHeader("MyRequestHeader"), "MyRequestHeaderValue");
       Assert.assertEquals(request.getParameter("MyParam"), "MyParamValue");
       
       final HttpServletResponse response = HttpServletRequestResponseContext.getResponse();
       assert(response != null);
       Assert.assertTrue(response.containsHeader("MyResponseHeader"));
       
       HttpServletRequestResponseContext.clearCurrent();
       
       Assert.assertNull(HttpServletRequestResponseContext.getRequest()); 
       Assert.assertNull(HttpServletRequestResponseContext.getResponse()); 
    }

}
