/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.httpclient;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.hc.client5.http.classic.methods.HttpGet;
import org.apache.hc.client5.http.protocol.HttpClientContext;
import org.apache.hc.core5.http.ClassicHttpRequest;
import org.apache.hc.core5.http.ClassicHttpResponse;
import org.apache.hc.core5.http.HttpHost;
import org.apache.hc.core5.http.HttpStatus;
import org.apache.hc.core5.http.ProtocolVersion;
import org.apache.hc.core5.http.io.HttpClientResponseHandler;
import org.apache.hc.core5.http.message.BasicClassicHttpResponse;
import org.apache.hc.core5.http.protocol.HttpContext;
import org.apache.hc.core5.io.CloseMode;
import org.testng.Assert;
import org.testng.annotations.BeforeClass;
import org.testng.annotations.BeforeMethod;
import org.testng.annotations.Test;

import net.shibboleth.shared.collection.CollectionSupport;

@SuppressWarnings("javadoc")
public class ContextHandlingHttpClientTest {
    
    public static final ClassicHttpResponse STATIC_RESPONSE_HTTP = 
            new MockClassicHttpResponse(new ProtocolVersion("HTTP", 1, 1), HttpStatus.SC_OK, "OK");
        
    public static final Object STATIC_RESPONSE_HANDLER = new Object();
    
    private ContextHandlingHttpClient client;
    
    private TestContextHandler staticOne, staticTwo, staticThree;
    private TestContextHandler dynamicOne, dynamicTwo, dynamicThree;
    
    private HttpClientContext context;
    
    private ClassicHttpRequest request;
    private HttpHost target;
    private HttpClientResponseHandler<Object> responseHandler = new MockResponseHandler();
    
    @BeforeClass
    public void setupClass() {
        staticOne = new TestContextHandler("static-1");
        staticTwo = new TestContextHandler("static-2");
        staticThree = new TestContextHandler("static-3");
        dynamicOne = new TestContextHandler("dynamic-1");
        dynamicTwo = new TestContextHandler("dynamic-2");
        dynamicThree = new TestContextHandler("dynamic-3");
        
    }
    
    @BeforeMethod
    public void setupMethod() {
        request = new HttpGet("/test");
        target = new HttpHost("test.example.edu");
    }
    
    @Test
    public void testNoHandlers() throws IOException {
        client = new ContextHandlingHttpClient(new MockHttpClient());
        context = HttpClientContext.create();
        assert context!=null;
        
        //Non-context execute methods
        Assert.assertSame(client.execute(request), STATIC_RESPONSE_HTTP);
        Assert.assertSame(client.execute(request, responseHandler), STATIC_RESPONSE_HANDLER);
        Assert.assertSame(client.execute(target, request), STATIC_RESPONSE_HTTP);
        Assert.assertSame(client.execute(target, request, responseHandler), STATIC_RESPONSE_HANDLER);
        
        //Context execute methods
        Assert.assertSame(client.execute(request, context), STATIC_RESPONSE_HTTP);
        Assert.assertSame(client.execute(request, context, responseHandler), STATIC_RESPONSE_HANDLER);
        Assert.assertSame(client.execute(target, request, context), STATIC_RESPONSE_HTTP);
        Assert.assertSame(client.execute(target, request, context, responseHandler), STATIC_RESPONSE_HANDLER);
    }
    
    @Test
    public void testStaticOnly() throws IOException {
        client = new ContextHandlingHttpClient(new MockHttpClient(), CollectionSupport.listOf(staticOne, staticTwo, staticThree));
        
        List<String> control = List.of(
                "before-static-1",
                "before-static-2",
                "before-static-3",
                "after-static-3",
                "after-static-2",
                "after-static-1"
                );
        
        context = HttpClientContext.create();
        assert context!=null;
        Assert.assertSame(client.execute(request, context), STATIC_RESPONSE_HTTP);
        Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        
        context = HttpClientContext.create();
        assert context!=null;
        Assert.assertSame(client.execute(request, context, responseHandler), STATIC_RESPONSE_HANDLER);
        Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        
        context = HttpClientContext.create();
        assert context!=null;
        Assert.assertSame(client.execute(target, request, context), STATIC_RESPONSE_HTTP);
        Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        
        context = HttpClientContext.create();
        Assert.assertSame(client.execute(target, request, context, responseHandler), STATIC_RESPONSE_HANDLER);
        Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
    }
    
    @Test
    public void testDynamicOnly() throws IOException {
        client = new ContextHandlingHttpClient(new MockHttpClient());
        
        List<String> control = List.of(
                "before-dynamic-1",
                "before-dynamic-2",
                "before-dynamic-3",
                "after-dynamic-3",
                "after-dynamic-2",
                "after-dynamic-1"
                );
        
        List<HttpClientContextHandler> handlers = List.of(dynamicOne, dynamicTwo, dynamicThree);
        
        context = HttpClientContext.create();
        assert context!=null;
        HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
        Assert.assertSame(client.execute(request, context), STATIC_RESPONSE_HTTP);
        Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        
        context = HttpClientContext.create();
        assert context!=null;
        HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
        Assert.assertSame(client.execute(request, context, responseHandler), STATIC_RESPONSE_HANDLER);
        Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        
        context = HttpClientContext.create();
        assert context!=null;
        HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
        Assert.assertSame(client.execute(target, request, context), STATIC_RESPONSE_HTTP);
        Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        
        context = HttpClientContext.create();
        assert context!=null;
        HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
        Assert.assertSame(client.execute(target, request, context, responseHandler), STATIC_RESPONSE_HANDLER);
        Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
    }
    
    @Test
    public void testStaticAndDynamic() throws IOException {
        client = new ContextHandlingHttpClient(new MockHttpClient(), CollectionSupport.listOf(staticOne, staticTwo, staticThree));
        
        List<String> control = List.of(
                "before-static-1",
                "before-static-2",
                "before-static-3",
                "before-dynamic-1",
                "before-dynamic-2",
                "before-dynamic-3",
                "after-dynamic-3",
                "after-dynamic-2",
                "after-dynamic-1",
                "after-static-3",
                "after-static-2",
                "after-static-1"
                );
        
        List<HttpClientContextHandler> handlers = List.of(dynamicOne, dynamicTwo, dynamicThree);
        
        context = HttpClientContext.create();
        assert context!=null;
        HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
        Assert.assertSame(client.execute(request, context), STATIC_RESPONSE_HTTP);
        Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        
        context = HttpClientContext.create();
        assert context!=null;
        HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
        Assert.assertSame(client.execute(request, context, responseHandler), STATIC_RESPONSE_HANDLER);
        Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        
        context = HttpClientContext.create();
        assert context!=null;
        HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
        Assert.assertSame(client.execute(target, request, context), STATIC_RESPONSE_HTTP);
        Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        
        context = HttpClientContext.create();
        assert context!=null;
        HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
        Assert.assertSame(client.execute(target, request, context, responseHandler), STATIC_RESPONSE_HANDLER);
        Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
    }
    
    @Test
    public void testWrappedClientThrowsIOException() throws IOException {
        IOException error = new IOException();
        client = new ContextHandlingHttpClient(new MockHttpClient(error), CollectionSupport.listOf(staticOne, staticTwo, staticThree));
        
        List<String> control = List.of(
                "before-static-1",
                "before-static-2",
                "before-static-3",
                "before-dynamic-1",
                "before-dynamic-2",
                "before-dynamic-3",
                "after-dynamic-3",
                "after-dynamic-2",
                "after-dynamic-1",
                "after-static-3",
                "after-static-2",
                "after-static-1"
                );
        
        List<HttpClientContextHandler> handlers = List.of(dynamicOne, dynamicTwo, dynamicThree);
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(request, context);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertSame(e, error);
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(request, context, responseHandler);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertSame(e, error);
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(target, request, context);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertSame(e, error);
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(target, request, context, responseHandler);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertSame(e, error);
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
    }
    
    @Test
    public void testWrappedClientThrowsRuntimeException() throws IOException {
        RuntimeException error = new RuntimeException();
        client = new ContextHandlingHttpClient(new MockHttpClient(error), CollectionSupport.listOf(staticOne, staticTwo, staticThree));
        
        List<String> control = List.of(
                "before-static-1",
                "before-static-2",
                "before-static-3",
                "before-dynamic-1",
                "before-dynamic-2",
                "before-dynamic-3",
                "after-dynamic-3",
                "after-dynamic-2",
                "after-dynamic-1",
                "after-static-3",
                "after-static-2",
                "after-static-1"
                );
        
        List<HttpClientContextHandler> handlers = List.of(dynamicOne, dynamicTwo, dynamicThree);
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(request, context);
            Assert.fail("Wrapped client should have thrown");
        } catch (RuntimeException e) {
            Assert.assertSame(e, error);
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(request, context, responseHandler);
            Assert.fail("Wrapped client should have thrown");
        } catch (RuntimeException e) {
            Assert.assertSame(e, error);
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(target, request, context);
            Assert.fail("Wrapped client should have thrown");
        } catch (RuntimeException e) {
            Assert.assertSame(e, error);
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(target, request, context, responseHandler);
            Assert.fail("Wrapped client should have thrown");
        } catch (RuntimeException e) {
            Assert.assertSame(e, error);
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
    }
    
    @Test
    public void testWrappedClientThrowsError() throws IOException {
        Error error = new Error();
        client = new ContextHandlingHttpClient(new MockHttpClient(error), CollectionSupport.listOf(staticOne, staticTwo, staticThree));
        
        List<String> control = List.of(
                "before-static-1",
                "before-static-2",
                "before-static-3",
                "before-dynamic-1",
                "before-dynamic-2",
                "before-dynamic-3",
                "after-dynamic-3",
                "after-dynamic-2",
                "after-dynamic-1",
                "after-static-3",
                "after-static-2",
                "after-static-1"
                );
        
        List<HttpClientContextHandler> handlers = CollectionSupport.listOf(dynamicOne, dynamicTwo, dynamicThree);
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(request, context);
            Assert.fail("Wrapped client should have thrown");
        } catch (Error e) {
            Assert.assertSame(e, error);
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(request, context, responseHandler);
            Assert.fail("Wrapped client should have thrown");
        } catch (Error e) {
            Assert.assertSame(e, error);
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(target, request, context);
            Assert.fail("Wrapped client should have thrown");
        } catch (Error e) {
            Assert.assertSame(e, error);
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(target, request, context, responseHandler);
            Assert.fail("Wrapped client should have thrown");
        } catch (Error e) {
            Assert.assertSame(e, error);
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
    }
     
    @Test
    public void testSingleStaticHandlerInvokeBeforeThrowsIOException() throws IOException {
        IOException error = new IOException();
        client = new ContextHandlingHttpClient(new MockHttpClient(), 
                CollectionSupport.listOf(staticOne, new TestContextHandler("static-2", error, null), staticThree));
        
        List<String> control = List.of(
                "before-static-1",
                "before-static-2",
                "before-static-3",
                "before-dynamic-1",
                "before-dynamic-2",
                "before-dynamic-3",
                "after-dynamic-3",
                "after-dynamic-2",
                "after-dynamic-1",
                "after-static-3",
                "after-static-2",
                "after-static-1"
                );
        
        List<HttpClientContextHandler> handlers = List.of(dynamicOne, dynamicTwo, dynamicThree);
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(request, context);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertSame(e, error);
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(request, context, responseHandler);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertSame(e, error);
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(target, request, context);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertSame(e, error);
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(target, request, context, responseHandler);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertSame(e, error);
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
    }
    
    @Test
    public void testSingleStaticHandlerInvokeAfterThrowsIOException() throws IOException {
        IOException error = new IOException();
        client = new ContextHandlingHttpClient(new MockHttpClient(), 
                CollectionSupport.listOf(staticOne, new TestContextHandler("static-2", null, error), staticThree));
        
        List<String> control = List.of(
                "before-static-1",
                "before-static-2",
                "before-static-3",
                "before-dynamic-1",
                "before-dynamic-2",
                "before-dynamic-3",
                "after-dynamic-3",
                "after-dynamic-2",
                "after-dynamic-1",
                "after-static-3",
                "after-static-2",
                "after-static-1"
                );
        
        List<HttpClientContextHandler> handlers = List.of(dynamicOne, dynamicTwo, dynamicThree);
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(request, context);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertSame(e, error);
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(request, context, responseHandler);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertSame(e, error);
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(target, request, context);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertSame(e, error);
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(target, request, context, responseHandler);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertSame(e, error);
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
    }
    
    @Test
    public void testSingleStaticHandlerInvokeBeforeThrowsRuntimeException() throws IOException {
        RuntimeException error = new RuntimeException();
        client = new ContextHandlingHttpClient(new MockHttpClient(), 
                CollectionSupport.listOf(staticOne, new TestContextHandler("static-2", error, null), staticThree));
        
        List<String> control = List.of(
                "before-static-1",
                "before-static-2",
                "before-static-3",
                "before-dynamic-1",
                "before-dynamic-2",
                "before-dynamic-3",
                "after-dynamic-3",
                "after-dynamic-2",
                "after-dynamic-1",
                "after-static-3",
                "after-static-2",
                "after-static-1"
                );
        
        List<HttpClientContextHandler> handlers = List.of(dynamicOne, dynamicTwo, dynamicThree);
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(request, context);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertSame(e.getCause(), error);
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(request, context, responseHandler);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertSame(e.getCause(), error);
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(target, request, context);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertSame(e.getCause(), error);
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(target, request, context, responseHandler);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertSame(e.getCause(), error);
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
    }
    
    @Test
    public void testSingleStaticHandlerInvokeAfterThrowsRuntimeException() throws IOException {
        RuntimeException error = new RuntimeException();
        client = new ContextHandlingHttpClient(new MockHttpClient(), 
                CollectionSupport.listOf(staticOne, new TestContextHandler("static-2", null, error), staticThree));
        
        List<String> control = List.of(
                "before-static-1",
                "before-static-2",
                "before-static-3",
                "before-dynamic-1",
                "before-dynamic-2",
                "before-dynamic-3",
                "after-dynamic-3",
                "after-dynamic-2",
                "after-dynamic-1",
                "after-static-3",
                "after-static-2",
                "after-static-1"
                );
        
        List<HttpClientContextHandler> handlers = List.of(dynamicOne, dynamicTwo, dynamicThree);
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(request, context);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertSame(e.getCause(), error);
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(request, context, responseHandler);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertSame(e.getCause(), error);
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(target, request, context);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertSame(e.getCause(), error);
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(target, request, context, responseHandler);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertSame(e.getCause(), error);
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
    }
    
    @Test
    public void testMultipleStaticHandlersInvokeBeforeThrowIOException() throws IOException {
        IOException error1 = new IOException();
        IOException error3 = new IOException();
        client = new ContextHandlingHttpClient(new MockHttpClient(), 
                CollectionSupport.listOf( new TestContextHandler("static-1", error1, null), staticTwo, new TestContextHandler("static-3", error3, null)));
        
        List<String> control = List.of(
                "before-static-1",
                "before-static-2",
                "before-static-3",
                "before-dynamic-1",
                "before-dynamic-2",
                "before-dynamic-3",
                "after-dynamic-3",
                "after-dynamic-2",
                "after-dynamic-1",
                "after-static-3",
                "after-static-2",
                "after-static-1"
                );
        
        List<HttpClientContextHandler> handlers = List.of(dynamicOne, dynamicTwo, dynamicThree);
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(request, context);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertNotSame(e, error1);
            Assert.assertNotSame(e, error3);
            Assert.assertEquals(e.getSuppressed().length, 2);
            Assert.assertTrue(Arrays.asList(e.getSuppressed()).containsAll(List.of(error1, error3)));
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(request, context, responseHandler);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertNotSame(e, error1);
            Assert.assertNotSame(e, error3);
            Assert.assertEquals(e.getSuppressed().length, 2);
            Assert.assertTrue(Arrays.asList(e.getSuppressed()).containsAll(List.of(error1, error3)));
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(target, request, context);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertNotSame(e, error1);
            Assert.assertNotSame(e, error3);
            Assert.assertEquals(e.getSuppressed().length, 2);
            Assert.assertTrue(Arrays.asList(e.getSuppressed()).containsAll(List.of(error1, error3)));
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(target, request, context, responseHandler);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertNotSame(e, error1);
            Assert.assertNotSame(e, error3);
            Assert.assertEquals(e.getSuppressed().length, 2);
            Assert.assertTrue(Arrays.asList(e.getSuppressed()).containsAll(List.of(error1, error3)));
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
    }
    
    @Test
    public void testMultipleStaticHandlersInvokeBeforeThrowRuntimeException() throws IOException {
        RuntimeException error1 = new RuntimeException();
        RuntimeException error3 = new RuntimeException();
        client = new ContextHandlingHttpClient(new MockHttpClient(), 
                 CollectionSupport.listOf( new TestContextHandler("static-1", error1, null), staticTwo, new TestContextHandler("static-3", error3, null)));
        
        List<String> control = List.of(
                "before-static-1",
                "before-static-2",
                "before-static-3",
                "before-dynamic-1",
                "before-dynamic-2",
                "before-dynamic-3",
                "after-dynamic-3",
                "after-dynamic-2",
                "after-dynamic-1",
                "after-static-3",
                "after-static-2",
                "after-static-1"
                );
        
        List<HttpClientContextHandler> handlers = List.of(dynamicOne, dynamicTwo, dynamicThree);
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(request, context);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertNotSame(e, error1);
            Assert.assertNotSame(e, error3);
            Assert.assertEquals(e.getSuppressed().length, 2);
            Assert.assertTrue(Arrays.asList(e.getSuppressed()).containsAll(List.of(error1, error3)));
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(request, context, responseHandler);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertNotSame(e, error1);
            Assert.assertNotSame(e, error3);
            Assert.assertEquals(e.getSuppressed().length, 2);
            Assert.assertTrue(Arrays.asList(e.getSuppressed()).containsAll(List.of(error1, error3)));
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(target, request, context);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertNotSame(e, error1);
            Assert.assertNotSame(e, error3);
            Assert.assertEquals(e.getSuppressed().length, 2);
            Assert.assertTrue(Arrays.asList(e.getSuppressed()).containsAll(List.of(error1, error3)));
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(target, request, context, responseHandler);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertNotSame(e, error1);
            Assert.assertNotSame(e, error3);
            Assert.assertEquals(e.getSuppressed().length, 2);
            Assert.assertTrue(Arrays.asList(e.getSuppressed()).containsAll(List.of(error1, error3)));
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
    }
    
    
    @Test
    public void testMultipleStaticHandlersInvokeAfterThrowIOException() throws IOException {
        IOException error1 = new IOException();
        IOException error3 = new IOException();
        client = new ContextHandlingHttpClient(new MockHttpClient(), 
                CollectionSupport.listOf( new TestContextHandler("static-1", null, error1), staticTwo, new TestContextHandler("static-3", null, error3)));
        
        List<String> control = List.of(
                "before-static-1",
                "before-static-2",
                "before-static-3",
                "before-dynamic-1",
                "before-dynamic-2",
                "before-dynamic-3",
                "after-dynamic-3",
                "after-dynamic-2",
                "after-dynamic-1",
                "after-static-3",
                "after-static-2",
                "after-static-1"
                );
        
        List<HttpClientContextHandler> handlers = List.of(dynamicOne, dynamicTwo, dynamicThree);
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(request, context);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertNotSame(e, error1);
            Assert.assertNotSame(e, error3);
            Assert.assertEquals(e.getSuppressed().length, 2);
            Assert.assertTrue(Arrays.asList(e.getSuppressed()).containsAll(List.of(error1, error3)));
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(request, context, responseHandler);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertNotSame(e, error1);
            Assert.assertNotSame(e, error3);
            Assert.assertEquals(e.getSuppressed().length, 2);
            Assert.assertTrue(Arrays.asList(e.getSuppressed()).containsAll(List.of(error1, error3)));
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(target, request, context);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertNotSame(e, error1);
            Assert.assertNotSame(e, error3);
            Assert.assertEquals(e.getSuppressed().length, 2);
            Assert.assertTrue(Arrays.asList(e.getSuppressed()).containsAll(List.of(error1, error3)));
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(target, request, context, responseHandler);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertNotSame(e, error1);
            Assert.assertNotSame(e, error3);
            Assert.assertEquals(e.getSuppressed().length, 2);
            Assert.assertTrue(Arrays.asList(e.getSuppressed()).containsAll(List.of(error1, error3)));
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
    }
    
    @Test
    public void testMultipleStaticHandlersInvokeAfterThrowRuntimeException() throws IOException {
        RuntimeException error1 = new RuntimeException();
        RuntimeException error3 = new RuntimeException();
        client = new ContextHandlingHttpClient(new MockHttpClient(), 
                CollectionSupport.listOf( new TestContextHandler("static-1", null, error1), staticTwo, new TestContextHandler("static-3", null, error3)));
        
        List<String> control = List.of(
                "before-static-1",
                "before-static-2",
                "before-static-3",
                "before-dynamic-1",
                "before-dynamic-2",
                "before-dynamic-3",
                "after-dynamic-3",
                "after-dynamic-2",
                "after-dynamic-1",
                "after-static-3",
                "after-static-2",
                "after-static-1"
                );
        
        List<HttpClientContextHandler> handlers = List.of(dynamicOne, dynamicTwo, dynamicThree);
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(request, context);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertNotSame(e, error1);
            Assert.assertNotSame(e, error3);
            Assert.assertEquals(e.getSuppressed().length, 2);
            Assert.assertTrue(Arrays.asList(e.getSuppressed()).containsAll(List.of(error1, error3)));
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(request, context, responseHandler);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertNotSame(e, error1);
            Assert.assertNotSame(e, error3);
            Assert.assertEquals(e.getSuppressed().length, 2);
            Assert.assertTrue(Arrays.asList(e.getSuppressed()).containsAll(List.of(error1, error3)));
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(target, request, context);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertNotSame(e, error1);
            Assert.assertNotSame(e, error3);
            Assert.assertEquals(e.getSuppressed().length, 2);
            Assert.assertTrue(Arrays.asList(e.getSuppressed()).containsAll(List.of(error1, error3)));
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(target, request, context, responseHandler);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertNotSame(e, error1);
            Assert.assertNotSame(e, error3);
            Assert.assertEquals(e.getSuppressed().length, 2);
            Assert.assertTrue(Arrays.asList(e.getSuppressed()).containsAll(List.of(error1, error3)));
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
    }
    
    
    @Test
    public void testSingleDynamicHandlerInvokeBeforeThrowsIOException() throws IOException {
        IOException error = new IOException();
        client = new ContextHandlingHttpClient(new MockHttpClient(), 
                CollectionSupport.listOf(staticOne, staticTwo, staticThree));
        
        @Nonnull List<String> control = CollectionSupport.listOf(
                "before-static-1",
                "before-static-2",
                "before-static-3",
                "before-dynamic-1",
                "before-dynamic-2",
                "before-dynamic-3",
                "after-dynamic-3",
                "after-dynamic-2",
                "after-dynamic-1",
                "after-static-3",
                "after-static-2",
                "after-static-1"
                );
        
        List<HttpClientContextHandler> handlers = List.of(dynamicOne, new TestContextHandler("dynamic-2", error, null), dynamicThree);
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(request, context);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertSame(e, error);
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(request, context, responseHandler);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertSame(e, error);
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(target, request, context);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertSame(e, error);
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(target, request, context, responseHandler);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertSame(e, error);
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
    }
    
    @Test
    public void testSingleDynamicHandlerInvokeAfterThrowsIOException() throws IOException {
        IOException error = new IOException();
        client = new ContextHandlingHttpClient(new MockHttpClient(), 
                CollectionSupport.listOf(staticOne, staticTwo, staticThree));
        
        List<String> control = CollectionSupport.listOf(
                "before-static-1",
                "before-static-2",
                "before-static-3",
                "before-dynamic-1",
                "before-dynamic-2",
                "before-dynamic-3",
                "after-dynamic-3",
                "after-dynamic-2",
                "after-dynamic-1",
                "after-static-3",
                "after-static-2",
                "after-static-1"
                );
        
        List<HttpClientContextHandler> handlers = List.of(dynamicOne, new TestContextHandler("dynamic-2", null, error), dynamicThree);
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(request, context);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertSame(e, error);
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(request, context, responseHandler);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertSame(e, error);
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(target, request, context);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertSame(e, error);
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(target, request, context, responseHandler);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertSame(e, error);
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
    }
    
    
    @Test
    public void testSingleDynamicHandlerInvokeBeforeThrowsRuntimeException() throws IOException {
        RuntimeException error = new RuntimeException();
        client = new ContextHandlingHttpClient(new MockHttpClient(), 
                CollectionSupport.listOf(staticOne, staticTwo, staticThree));
        
        List<String> control = CollectionSupport.listOf(
                "before-static-1",
                "before-static-2",
                "before-static-3",
                "before-dynamic-1",
                "before-dynamic-2",
                "before-dynamic-3",
                "after-dynamic-3",
                "after-dynamic-2",
                "after-dynamic-1",
                "after-static-3",
                "after-static-2",
                "after-static-1"
                );
        
        List<HttpClientContextHandler> handlers = List.of(dynamicOne, new TestContextHandler("dynamic-2", error, null), dynamicThree);
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(request, context);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertSame(e.getCause(), error);
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(request, context, responseHandler);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertSame(e.getCause(), error);
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(target, request, context);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertSame(e.getCause(), error);
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(target, request, context, responseHandler);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertSame(e.getCause(), error);
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
    }
    
    @Test
    public void testSingleDynamicHandlerInvokeAfterThrowsRuntimeException() throws IOException {
        RuntimeException error = new RuntimeException();
        client = new ContextHandlingHttpClient(new MockHttpClient(), 
                CollectionSupport.listOf(staticOne, staticTwo, staticThree));
        
        List<String> control = CollectionSupport.listOf(
                "before-static-1",
                "before-static-2",
                "before-static-3",
                "before-dynamic-1",
                "before-dynamic-2",
                "before-dynamic-3",
                "after-dynamic-3",
                "after-dynamic-2",
                "after-dynamic-1",
                "after-static-3",
                "after-static-2",
                "after-static-1"
                );
        
        List<HttpClientContextHandler> handlers = List.of(dynamicOne, new TestContextHandler("dynamic-2", null, error), dynamicThree);
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(request, context);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertSame(e.getCause(), error);
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(request, context, responseHandler);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertSame(e.getCause(), error);
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(target, request, context);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertSame(e.getCause(), error);
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(target, request, context, responseHandler);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertSame(e.getCause(), error);
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
    }
    
    
    @Test
    public void testMultipleDynamicHandlersInvokeBeforeThrowIOException() throws IOException {
        IOException error1 = new IOException();
        IOException error3 = new IOException();
        client = new ContextHandlingHttpClient(new MockHttpClient(), CollectionSupport.listOf(staticOne, staticTwo, staticThree));
        
        List<String> control = CollectionSupport.listOf(
                "before-static-1",
                "before-static-2",
                "before-static-3",
                "before-dynamic-1",
                "before-dynamic-2",
                "before-dynamic-3",
                "after-dynamic-3",
                "after-dynamic-2",
                "after-dynamic-1",
                "after-static-3",
                "after-static-2",
                "after-static-1"
                );
        
        List<HttpClientContextHandler> handlers = 
                List.of(new TestContextHandler("dynamic-1", error1, null), dynamicTwo, new TestContextHandler("dynamic-3", error3, null));
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(request, context);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertNotSame(e, error1);
            Assert.assertNotSame(e, error3);
            Assert.assertEquals(e.getSuppressed().length, 2);
            Assert.assertTrue(Arrays.asList(e.getSuppressed()).containsAll(List.of(error1, error3)));
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(request, context, responseHandler);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertNotSame(e, error1);
            Assert.assertNotSame(e, error3);
            Assert.assertEquals(e.getSuppressed().length, 2);
            Assert.assertTrue(Arrays.asList(e.getSuppressed()).containsAll(List.of(error1, error3)));
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(target, request, context);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertNotSame(e, error1);
            Assert.assertNotSame(e, error3);
            Assert.assertEquals(e.getSuppressed().length, 2);
            Assert.assertTrue(Arrays.asList(e.getSuppressed()).containsAll(List.of(error1, error3)));
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(target, request, context, responseHandler);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertNotSame(e, error1);
            Assert.assertNotSame(e, error3);
            Assert.assertEquals(e.getSuppressed().length, 2);
            Assert.assertTrue(Arrays.asList(e.getSuppressed()).containsAll(List.of(error1, error3)));
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
    }
    
    @Test
    public void testMultipleDynamicHandlersInvokeBeforeThrowRuntimeException() throws IOException {
        RuntimeException error1 = new RuntimeException();
        RuntimeException error3 = new RuntimeException();
        client = new ContextHandlingHttpClient(new MockHttpClient(), CollectionSupport.listOf(staticOne, staticTwo, staticThree));
        
        List<String> control = CollectionSupport.listOf(
                "before-static-1",
                "before-static-2",
                "before-static-3",
                "before-dynamic-1",
                "before-dynamic-2",
                "before-dynamic-3",
                "after-dynamic-3",
                "after-dynamic-2",
                "after-dynamic-1",
                "after-static-3",
                "after-static-2",
                "after-static-1"
                );
        
        List<HttpClientContextHandler> handlers = 
                List.of(new TestContextHandler("dynamic-1", error1, null), dynamicTwo, new TestContextHandler("dynamic-3", error3, null));
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(request, context);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertNotSame(e, error1);
            Assert.assertNotSame(e, error3);
            Assert.assertEquals(e.getSuppressed().length, 2);
            Assert.assertTrue(Arrays.asList(e.getSuppressed()).containsAll(List.of(error1, error3)));
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(request, context, responseHandler);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertNotSame(e, error1);
            Assert.assertNotSame(e, error3);
            Assert.assertEquals(e.getSuppressed().length, 2);
            Assert.assertTrue(Arrays.asList(e.getSuppressed()).containsAll(List.of(error1, error3)));
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(target, request, context);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertNotSame(e, error1);
            Assert.assertNotSame(e, error3);
            Assert.assertEquals(e.getSuppressed().length, 2);
            Assert.assertTrue(Arrays.asList(e.getSuppressed()).containsAll(List.of(error1, error3)));
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(target, request, context, responseHandler);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertNotSame(e, error1);
            Assert.assertNotSame(e, error3);
            Assert.assertEquals(e.getSuppressed().length, 2);
            Assert.assertTrue(Arrays.asList(e.getSuppressed()).containsAll(List.of(error1, error3)));
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
    }
    
    @Test
    public void testMultipleDynamicHandlersInvokeAfterThrowIOException() throws IOException {
        IOException error1 = new IOException();
        IOException error3 = new IOException();
        client = new ContextHandlingHttpClient(new MockHttpClient(), CollectionSupport.listOf(staticOne, staticTwo, staticThree));
        
        List<String> control = CollectionSupport.listOf(
                "before-static-1",
                "before-static-2",
                "before-static-3",
                "before-dynamic-1",
                "before-dynamic-2",
                "before-dynamic-3",
                "after-dynamic-3",
                "after-dynamic-2",
                "after-dynamic-1",
                "after-static-3",
                "after-static-2",
                "after-static-1"
                );
        
        List<HttpClientContextHandler> handlers = 
                List.of(new TestContextHandler("dynamic-1", null, error1), dynamicTwo, new TestContextHandler("dynamic-3", null, error3));
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(request, context);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertNotSame(e, error1);
            Assert.assertNotSame(e, error3);
            Assert.assertEquals(e.getSuppressed().length, 2);
            Assert.assertTrue(Arrays.asList(e.getSuppressed()).containsAll(List.of(error1, error3)));
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(request, context, responseHandler);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertNotSame(e, error1);
            Assert.assertNotSame(e, error3);
            Assert.assertEquals(e.getSuppressed().length, 2);
            Assert.assertTrue(Arrays.asList(e.getSuppressed()).containsAll(List.of(error1, error3)));
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(target, request, context);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertNotSame(e, error1);
            Assert.assertNotSame(e, error3);
            Assert.assertEquals(e.getSuppressed().length, 2);
            Assert.assertTrue(Arrays.asList(e.getSuppressed()).containsAll(List.of(error1, error3)));
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(target, request, context, responseHandler);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertNotSame(e, error1);
            Assert.assertNotSame(e, error3);
            Assert.assertEquals(e.getSuppressed().length, 2);
            Assert.assertTrue(Arrays.asList(e.getSuppressed()).containsAll(List.of(error1, error3)));
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
    }
    
    @Test
    public void testMultipleDynamicHandlersInvokeAfterThrowRuntimeException() throws IOException {
        RuntimeException error1 = new RuntimeException();
        RuntimeException error3 = new RuntimeException();
        client = new ContextHandlingHttpClient(new MockHttpClient(), CollectionSupport.listOf(staticOne, staticTwo, staticThree));
        
        List<String> control = CollectionSupport.listOf(
                "before-static-1",
                "before-static-2",
                "before-static-3",
                "before-dynamic-1",
                "before-dynamic-2",
                "before-dynamic-3",
                "after-dynamic-3",
                "after-dynamic-2",
                "after-dynamic-1",
                "after-static-3",
                "after-static-2",
                "after-static-1"
                );
        
        List<HttpClientContextHandler> handlers = 
                List.of(new TestContextHandler("dynamic-1", null, error1), dynamicTwo, new TestContextHandler("dynamic-3", null, error3));
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(request, context);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertNotSame(e, error1);
            Assert.assertNotSame(e, error3);
            Assert.assertEquals(e.getSuppressed().length, 2);
            Assert.assertTrue(Arrays.asList(e.getSuppressed()).containsAll(List.of(error1, error3)));
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(request, context, responseHandler);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertNotSame(e, error1);
            Assert.assertNotSame(e, error3);
            Assert.assertEquals(e.getSuppressed().length, 2);
            Assert.assertTrue(Arrays.asList(e.getSuppressed()).containsAll(List.of(error1, error3)));
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(target, request, context);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertNotSame(e, error1);
            Assert.assertNotSame(e, error3);
            Assert.assertEquals(e.getSuppressed().length, 2);
            Assert.assertTrue(Arrays.asList(e.getSuppressed()).containsAll(List.of(error1, error3)));
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(target, request, context, responseHandler);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertNotSame(e, error1);
            Assert.assertNotSame(e, error3);
            Assert.assertEquals(e.getSuppressed().length, 2);
            Assert.assertTrue(Arrays.asList(e.getSuppressed()).containsAll(List.of(error1, error3)));
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
    }
    
    
    @Test
    public void testStaticAndDynamicHandlersThrowIOException() throws IOException {
        IOException staticBeforeError = null;
        IOException dynamicAfterError = null;
        
        List<String> control = CollectionSupport.listOf(
                "before-static-1",
                "before-static-2",
                "before-static-3",
                "before-dynamic-1",
                "before-dynamic-2",
                "before-dynamic-3",
                "after-dynamic-3",
                "after-dynamic-2",
                "after-dynamic-1",
                "after-static-3",
                "after-static-2",
                "after-static-1"
                );
        
        List<HttpClientContextHandler> dynamicHandlers = null;
        
        staticBeforeError = new IOException();
        dynamicAfterError = new IOException();
        
        client = new ContextHandlingHttpClient(new MockHttpClient(), 
                CollectionSupport.listOf(staticOne, new TestContextHandler("static-2", staticBeforeError, null), staticThree));
        dynamicHandlers = CollectionSupport.listOf(dynamicOne, new TestContextHandler("dynamic-2", null, dynamicAfterError), dynamicThree);
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(dynamicHandlers);
            client.execute(request, context);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertSame(e, staticBeforeError);
            Assert.assertEquals(e.getSuppressed().length, 1);
            Assert.assertSame(e.getSuppressed()[0], dynamicAfterError);
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        staticBeforeError = new IOException();
        dynamicAfterError = new IOException();
        
        client = new ContextHandlingHttpClient(new MockHttpClient(), 
                CollectionSupport.listOf(staticOne, new TestContextHandler("static-2", staticBeforeError, null), staticThree));
        dynamicHandlers = List.of(dynamicOne, new TestContextHandler("dynamic-2", null, dynamicAfterError), dynamicThree);
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(dynamicHandlers);
            client.execute(request, context, responseHandler);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertSame(e, staticBeforeError);
            Assert.assertEquals(e.getSuppressed().length, 1);
            Assert.assertSame(e.getSuppressed()[0], dynamicAfterError);
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        staticBeforeError = new IOException();
        dynamicAfterError = new IOException();
        
        client = new ContextHandlingHttpClient(new MockHttpClient(), 
                CollectionSupport.listOf(staticOne, new TestContextHandler("static-2", staticBeforeError, null), staticThree));
        dynamicHandlers = CollectionSupport.listOf(dynamicOne, new TestContextHandler("dynamic-2", null, dynamicAfterError), dynamicThree);
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(dynamicHandlers);
            client.execute(target, request, context);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertSame(e, staticBeforeError);
            Assert.assertEquals(e.getSuppressed().length, 1);
            Assert.assertSame(e.getSuppressed()[0], dynamicAfterError);
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        staticBeforeError = new IOException();
        dynamicAfterError = new IOException();
        
        client = new ContextHandlingHttpClient(new MockHttpClient(), 
                CollectionSupport.listOf(staticOne, new TestContextHandler("static-2", staticBeforeError, null), staticThree));
        dynamicHandlers = CollectionSupport.listOf(dynamicOne, new TestContextHandler("dynamic-2", null, dynamicAfterError), dynamicThree);
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(dynamicHandlers);
            client.execute(target, request, context, responseHandler);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertSame(e, staticBeforeError);
            Assert.assertEquals(e.getSuppressed().length, 1);
            Assert.assertSame(e.getSuppressed()[0], dynamicAfterError);
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
    }
    
    @Test
    public void testStaticAndDynamicHandlersThrowRuntimeException() throws IOException {
        RuntimeException staticBeforeError = new RuntimeException();
        RuntimeException dynamicAfterError = new RuntimeException();
        client = new ContextHandlingHttpClient(new MockHttpClient(), 
                CollectionSupport.listOf(staticOne, new TestContextHandler("static-2", staticBeforeError, null), staticThree));
        
        List<String> control = CollectionSupport.listOf(
                "before-static-1",
                "before-static-2",
                "before-static-3",
                "before-dynamic-1",
                "before-dynamic-2",
                "before-dynamic-3",
                "after-dynamic-3",
                "after-dynamic-2",
                "after-dynamic-1",
                "after-static-3",
                "after-static-2",
                "after-static-1"
                );
        
        List<HttpClientContextHandler> handlers = List.of(dynamicOne, new TestContextHandler("dynamic-2", null, dynamicAfterError), dynamicThree);
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(request, context);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertSame(e.getCause(), staticBeforeError);
            Assert.assertEquals(e.getSuppressed().length, 1);
            Assert.assertSame(e.getSuppressed()[0].getCause(), dynamicAfterError);
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(request, context, responseHandler);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertSame(e.getCause(), staticBeforeError);
            Assert.assertEquals(e.getSuppressed().length, 1);
            Assert.assertSame(e.getSuppressed()[0].getCause(), dynamicAfterError);
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(target, request, context);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertSame(e.getCause(), staticBeforeError);
            Assert.assertEquals(e.getSuppressed().length, 1);
            Assert.assertSame(e.getSuppressed()[0].getCause(), dynamicAfterError);
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(handlers);
            client.execute(target, request, context, responseHandler);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertSame(e.getCause(), staticBeforeError);
            Assert.assertEquals(e.getSuppressed().length, 1);
            Assert.assertSame(e.getSuppressed()[0].getCause(), dynamicAfterError);
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
    }
    
    @Test
    public void testWrappedClientThrowsIOExceptionInvokeAfterThrowsIOException() throws IOException {
        IOException clientError = null;
        IOException dynamicAfterError = null;
        
        List<String> control = CollectionSupport.listOf(
                "before-static-1",
                "before-static-2",
                "before-static-3",
                "before-dynamic-1",
                "before-dynamic-2",
                "before-dynamic-3",
                "after-dynamic-3",
                "after-dynamic-2",
                "after-dynamic-1",
                "after-static-3",
                "after-static-2",
                "after-static-1"
                );
        
        List<HttpClientContextHandler> dynamicHandlers = null;
        
        clientError = new IOException();
        dynamicAfterError = new IOException();
        
        client = new ContextHandlingHttpClient(new MockHttpClient(clientError), 
                CollectionSupport.listOf(staticOne, staticTwo, staticThree));
        dynamicHandlers = List.of(dynamicOne, new TestContextHandler("dynamic-2", null, dynamicAfterError), dynamicThree);
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(dynamicHandlers);
            client.execute(request, context);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertSame(e, clientError);
            Assert.assertEquals(e.getSuppressed().length, 1);
            Assert.assertSame(e.getSuppressed()[0], dynamicAfterError);
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        clientError = new IOException();
        dynamicAfterError = new IOException();
        
        client = new ContextHandlingHttpClient(new MockHttpClient(clientError), 
                CollectionSupport.listOf(staticOne, staticTwo, staticThree));
        dynamicHandlers = List.of(dynamicOne, new TestContextHandler("dynamic-2", null, dynamicAfterError), dynamicThree);
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(dynamicHandlers);
            client.execute(request, context, responseHandler);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertSame(e, clientError);
            Assert.assertEquals(e.getSuppressed().length, 1);
            Assert.assertSame(e.getSuppressed()[0], dynamicAfterError);
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        clientError = new IOException();
        dynamicAfterError = new IOException();
        
        client = new ContextHandlingHttpClient(new MockHttpClient(clientError), 
                CollectionSupport.listOf(staticOne, staticTwo, staticThree));
        dynamicHandlers = List.of(dynamicOne, new TestContextHandler("dynamic-2", null, dynamicAfterError), dynamicThree);
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(dynamicHandlers);
            client.execute(target, request, context);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertSame(e, clientError);
            Assert.assertEquals(e.getSuppressed().length, 1);
            Assert.assertSame(e.getSuppressed()[0], dynamicAfterError);
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
        
        clientError = new IOException();
        dynamicAfterError = new IOException();
        
        client = new ContextHandlingHttpClient(new MockHttpClient(clientError), 
                CollectionSupport.listOf(staticOne, staticTwo, staticThree));
        dynamicHandlers = List.of(dynamicOne, new TestContextHandler("dynamic-2", null, dynamicAfterError), dynamicThree);
        
        try {
            context = HttpClientContext.create();
            assert context!=null;
            HttpClientSupport.getDynamicContextHandlerList(context).addAll(dynamicHandlers);
            client.execute(target, request, context, responseHandler);
            Assert.fail("Wrapped client should have thrown");
        } catch (IOException e) {
            Assert.assertSame(e, clientError);
            Assert.assertEquals(e.getSuppressed().length, 1);
            Assert.assertSame(e.getSuppressed()[0], dynamicAfterError);
            Assert.assertEquals(context.getAttribute(TestContextHandler.TEST_KEY), control);
        }
    }
     
    
    
    
    
    // Helpers
    
    private class TestContextHandler implements HttpClientContextHandler {
        
        public static final String TEST_KEY = "testKey";
        
        private String name;
        
        private Throwable invokeBeforeError;
        private Throwable invokeAfterError;
        
        public TestContextHandler(String instanceName) {
            name = instanceName;
        }
        
        public TestContextHandler(String instanceName, Throwable beforeError, Throwable afterError) {
            name = instanceName;
            invokeBeforeError = beforeError;
            invokeAfterError = afterError;
        }
        
        /** {@inheritDoc} */
        public void invokeBefore(@Nonnull HttpClientContext context, @Nonnull ClassicHttpRequest request) throws IOException {
            if (name != null) {
                addValue(context, "before-" + name);
            }
            ThrowableHelper.checkAndThrowError(invokeBeforeError);
        }

        /** {@inheritDoc} */
        public void invokeAfter(@Nonnull HttpClientContext context, @Nonnull ClassicHttpRequest request) throws IOException {
            if (name != null) {
                addValue(context, "after-" + name);
            }
            ThrowableHelper.checkAndThrowError(invokeAfterError);
        }
        
        private void addValue(HttpClientContext context, String value) {
            @SuppressWarnings("unchecked")
            List<String> attrib = context.getAttribute(TEST_KEY, List.class);
            if (attrib == null) {
                attrib = new ArrayList<>();
                context.setAttribute(TEST_KEY, attrib);
            }
            attrib.add(value);
        }
        
    }
    
    private static class MockHttpClient extends AbstractHttpClient {
        
        private Throwable error;
        
        public MockHttpClient() {
        }
        
        public MockHttpClient(final Throwable throwable) {
            error = throwable;
        }
        
        /** {@inheritDoc} */
        public void close() throws IOException {
            // nothing to do
        }

        /** {@inheritDoc} */
        public void close(CloseMode closeMode) {
           // nothing to do 
        }
        
        /** {@inheritDoc} */
        protected ClassicHttpResponse doExecute(@Nullable final HttpHost target, @Nonnull final ClassicHttpRequest request,
                @Nullable final HttpContext context) throws IOException {
            ThrowableHelper.checkAndThrowError(error);
            return STATIC_RESPONSE_HTTP;
        }

    }

    public static class MockResponseHandler implements HttpClientResponseHandler<Object> {

        public Object handleResponse(ClassicHttpResponse response) throws IOException {
            return STATIC_RESPONSE_HANDLER;
        }

    }
    
    public static class MockClassicHttpResponse extends BasicClassicHttpResponse {

        private static final long serialVersionUID = -3530926876971302045L;

        public MockClassicHttpResponse(ProtocolVersion ver, int code, String reason) {
            super(code, reason);
            this.setVersion(ver);
        }

        public void close() throws IOException {
            
        }
        
    }
    
    public static class ThrowableHelper {
        
        public static void checkAndThrowError(Throwable t) throws IOException {
            if (t != null) {
                if (IOException.class.isInstance(t)) {
                    throw IOException.class.cast(t);
                } 
                if (RuntimeException.class.isInstance(t)) {
                    throw RuntimeException.class.cast(t);
                } 
                if (Error.class.isInstance(t)) {
                    throw Error.class.cast(t);
                } 
            }
        }
        
    }

}
