/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package net.shibboleth.shared.httpclient;

import java.io.IOException;
import java.time.Duration;
import java.util.List;

import org.apache.hc.client5.http.classic.HttpClient;
import org.apache.hc.client5.http.protocol.HttpClientContext;
import org.apache.hc.core5.http.ClassicHttpRequest;
import org.testng.Assert;
import org.testng.annotations.Test;

import net.shibboleth.shared.logic.ConstraintViolationException;

@SuppressWarnings("javadoc")
public class HttpClientBuilderTest {
    
    // Default timeouts
    @Test public void JSPT66() throws Exception {
        final HttpClientBuilder builder = new HttpClientBuilder();
        
        // Check the defaults at the builder level
        Assert.assertEquals(builder.getConnectionTimeout(), Duration.ofSeconds(60));
        Assert.assertEquals(builder.getSocketTimeout(), Duration.ofSeconds(60));
        Assert.assertEquals(builder.getConnectionRequestTimeout(), Duration.ofSeconds(60));
        
        try {
            builder.setSocketTimeout(Duration.ofMillis(Long.MAX_VALUE));
            Assert.fail();
        } catch (final ConstraintViolationException e) {
            
        }
        
        // Just make sure we can create a client, too
        final HttpClient client = builder.buildClient();
        Assert.assertNotNull(client);
    }
    
    @Test
    public void testContextHandlingSupport() throws Exception {
        
        HttpClientContextHandler handler1 = new TestContextHandler();
        HttpClientContextHandler handler2 = new TestContextHandler();
        HttpClientContextHandler handler3 = new TestContextHandler();
        
        final HttpClientBuilder builder = new HttpClientBuilder();
        
        Assert.assertNotNull(builder.getStaticContextHandlers());
        Assert.assertTrue(builder.getStaticContextHandlers().isEmpty());
        
        builder.setStaticContextHandlers(List.of(handler1, handler2, handler3));
        Assert.assertEquals(builder.getStaticContextHandlers(), List.of(handler1, handler2, handler3));
        
        try {
            builder.getStaticContextHandlers().add(new TestContextHandler());
            Assert.fail("List should have been unmodifaible");
        } catch (UnsupportedOperationException e) {
            //expected
        }
        
        builder.resetDefaults();
        
        Assert.assertNotNull(builder.getStaticContextHandlers());
        Assert.assertTrue(builder.getStaticContextHandlers().isEmpty());
        
        final HttpClient client = builder.buildClient();
        Assert.assertNotNull(client);
        Assert.assertTrue(ContextHandlingHttpClient.class.isInstance(client));
        
    }
    
    //Helpers 
    
    public class TestContextHandler implements HttpClientContextHandler {

        /** {@inheritDoc} */
        public void invokeBefore(HttpClientContext context, ClassicHttpRequest request) throws IOException {
            
        }

        /** {@inheritDoc} */
        public void invokeAfter(HttpClientContext context, ClassicHttpRequest request) throws IOException {
            
        }
        
    }
     
    
}
