/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.net;

import javax.annotation.Nonnull;

import org.springframework.mock.web.MockHttpServletRequest;
import org.springframework.mock.web.MockHttpServletResponse;
import org.testng.Assert;
import org.testng.annotations.Test;

import jakarta.servlet.http.Cookie;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import net.shibboleth.shared.component.ComponentInitializationException;
import net.shibboleth.shared.primitive.NonnullSupplier;

/** {@link CookieManager} unit test. */
@SuppressWarnings("javadoc")
public class CookieManagerTest {

    @Test public void testInitFailure() {
        CookieManager cm = new CookieManager();
        try {
            cm.initialize();
            Assert.fail();
        } catch (ComponentInitializationException e) {
            
        }
    }

    @Test public void testInitSuccess() throws ComponentInitializationException {
        MockHttpServletRequest request = new MockHttpServletRequest();
        MockHttpServletResponse response = new MockHttpServletResponse();
        
        CookieManager cm = new CookieManager();
        cm.setHttpServletRequestSupplier(new NonnullSupplier<>() { @Nonnull public HttpServletRequest get() {return request;}});
        cm.setHttpServletResponseSupplier(new NonnullSupplier<>() { @Nonnull public HttpServletResponse get() {return response;}});
        cm.initialize();
    }

    @Test public void testCookieWithPath() throws ComponentInitializationException {
        MockHttpServletRequest request = new MockHttpServletRequest();
        MockHttpServletResponse response = new MockHttpServletResponse();
        
        CookieManager cm = new CookieManager();
        cm.setHttpServletRequestSupplier(new NonnullSupplier<>() { @Nonnull public HttpServletRequest get() {return request;}});
        cm.setHttpServletResponseSupplier(new NonnullSupplier<>() { @Nonnull public HttpServletResponse get() {return response;}});
        cm.setCookiePath("/idp");
        cm.initialize();

        cm.addCookie("foo", "bar");

        Cookie cookie = response.getCookie("foo");
        assert(cookie != null);
        Assert.assertEquals(cookie.getValue(), "bar");
        Assert.assertEquals(cookie.getPath(), "/idp");
        Assert.assertNull(cookie.getDomain());
        Assert.assertTrue(cookie.getSecure());
        Assert.assertEquals(cookie.getMaxAge(), -1);
    }

    @Test public void testCookieNoPath() throws ComponentInitializationException {
        MockHttpServletRequest request = new MockHttpServletRequest();
        request.setContextPath("/idp");
        MockHttpServletResponse response = new MockHttpServletResponse();
        
        CookieManager cm = new CookieManager();
        cm.setHttpServletRequestSupplier(new NonnullSupplier<>() { @Nonnull public HttpServletRequest get() {return request;}});
        cm.setHttpServletResponseSupplier(new NonnullSupplier<>() { @Nonnull public HttpServletResponse get() {return response;}});
        cm.initialize();
        
        cm.addCookie("foo", "bar");
        
        Cookie cookie = response.getCookie("foo");
        assert(cookie != null);
        Assert.assertEquals(cookie.getValue(), "bar");
        Assert.assertEquals(cookie.getPath(), "/idp");
        Assert.assertNull(cookie.getDomain());
        Assert.assertTrue(cookie.getSecure());
        Assert.assertEquals(cookie.getMaxAge(), -1);
    }

    @Test public void testCookieUnset() throws ComponentInitializationException {
        MockHttpServletRequest request = new MockHttpServletRequest();
        request.setContextPath("/idp");
        request.setCookies(new Cookie("foo", "bar"));
        MockHttpServletResponse response = new MockHttpServletResponse();
        
        CookieManager cm = new CookieManager();
        cm.setHttpServletRequestSupplier(new NonnullSupplier<>() { @Nonnull public HttpServletRequest get() {return request;}});
        cm.setHttpServletResponseSupplier(new NonnullSupplier<>() { @Nonnull public HttpServletResponse get() {return response;}});
        cm.initialize();
        
        cm.unsetCookie("foo");
        
        Cookie cookie = response.getCookie("foo");
        assert(cookie != null);
        Assert.assertNull(cookie.getValue());
        Assert.assertEquals(cookie.getPath(), "/idp");
        Assert.assertNull(cookie.getDomain());
        Assert.assertTrue(cookie.getSecure());
        Assert.assertEquals(cookie.getMaxAge(), 0);
    }
}
