/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.servlet.impl;

import org.springframework.mock.web.MockHttpServletRequest;
import org.springframework.mock.web.MockHttpServletResponse;
import org.testng.Assert;
import org.testng.annotations.AfterMethod;
import org.testng.annotations.BeforeMethod;
import org.testng.annotations.Test;

import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import net.shibboleth.shared.logic.ConstraintViolationException;

/**
 * Tests for {@link HttpServletRequestResponseContext}.
 */
public class HttpServletRequestResponseContextTest {
    private HttpServletRequest request;
    private HttpServletResponse response;
    
    @BeforeMethod
    public void setUp() {
        final MockHttpServletRequest mockRequest = new MockHttpServletRequest();
        mockRequest.setMethod("GET");
        mockRequest.setRequestURI("/foo");
        mockRequest.addHeader("MyRequestHeader", "MyRequestHeaderValue");
        mockRequest.addParameter("MyParam", "MyParamValue");
        request = mockRequest;
        
        final MockHttpServletResponse mockResponse = new MockHttpServletResponse();
        mockResponse.setContentType("text/html");
        mockResponse.setCharacterEncoding("UTF-8");
        mockResponse.setHeader("MyResponseHeader", "MyResponseHeaderValue");
        response = mockResponse;
    }
    
    @AfterMethod
    public void tearDown()  {
        HttpServletRequestResponseContext.clearCurrent();
    }
    
    @Test(expectedExceptions= {ConstraintViolationException.class, IllegalStateException.class} )
    public void testRequestNoLoad() {
        Assert.assertNull(HttpServletRequestResponseContext.getRequest()); 
        Assert.assertNull(HttpServletRequestResponseContext.getResponse()); 
        
        final ThreadLocalHttpServletRequestSupplier proxy = new ThreadLocalHttpServletRequestSupplier();
        proxy.get().getMethod();
    }
    
    @Test
    public void testRequest() {
        Assert.assertNull(HttpServletRequestResponseContext.getRequest()); 
        Assert.assertNull(HttpServletRequestResponseContext.getResponse()); 
        
        HttpServletRequestResponseContext.loadCurrent(request, response);
        
        final ThreadLocalHttpServletRequestSupplier proxy = new ThreadLocalHttpServletRequestSupplier();
        final HttpServletRequest request = proxy.get();
        Assert.assertEquals(request.getMethod(), "GET");
        Assert.assertEquals(request.getRequestURI(), "/foo");
        Assert.assertEquals(request.getHeader("MyRequestHeader"), "MyRequestHeaderValue");
        Assert.assertEquals(request.getParameter("MyParam"), "MyParamValue");
        
        HttpServletRequestResponseContext.clearCurrent();
        
        Assert.assertNull(HttpServletRequestResponseContext.getRequest()); 
        Assert.assertNull(HttpServletRequestResponseContext.getResponse()); 
        
    }

    
    @Test(expectedExceptions= {ConstraintViolationException.class, IllegalStateException.class} )
    public void testResponseNoLoad() {
        Assert.assertNull(HttpServletRequestResponseContext.getRequest()); 
        Assert.assertNull(HttpServletRequestResponseContext.getResponse()); 
        
        final ThreadLocalHttpServletResponseSupplier proxy = new ThreadLocalHttpServletResponseSupplier();
        proxy.get().getContentType();
    }
    
    @Test
    public void testResponse() {
        Assert.assertNull(HttpServletRequestResponseContext.getRequest()); 
        Assert.assertNull(HttpServletRequestResponseContext.getResponse()); 
        
        HttpServletRequestResponseContext.loadCurrent(request, response);
        
        final ThreadLocalHttpServletResponseSupplier proxy = new ThreadLocalHttpServletResponseSupplier();
        final HttpServletResponse response = proxy.get();
        Assert.assertEquals(response.getContentType(), "text/html;charset=UTF-8");
        Assert.assertEquals(response.getCharacterEncoding(), "UTF-8");
        Assert.assertTrue(response.containsHeader("MyResponseHeader"));
        
        HttpServletRequestResponseContext.clearCurrent();
        
        Assert.assertNull(HttpServletRequestResponseContext.getRequest()); 
        Assert.assertNull(HttpServletRequestResponseContext.getResponse()); 
        
    }

}