/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.servlet.impl;

import java.util.List;
import java.util.Locale.LanguageRange;

import org.springframework.mock.web.MockHttpServletRequest;
import org.springframework.mock.web.MockHttpServletResponse;
import org.testng.Assert;
import org.testng.annotations.Test;

import com.google.common.net.MediaType;

import net.shibboleth.shared.collection.CollectionSupport;
import net.shibboleth.shared.servlet.HttpServletSupport;


/** {@link HttpServletSupport} unit test. */
@SuppressWarnings("javadoc")
public class HttpServletSupportTest {

    @Test public void testAddNoCacheHeaders(){
        MockHttpServletResponse response = new MockHttpServletResponse();
        Assert.assertNull(response.getHeaderValue("Cache-control"));
        Assert.assertNull(response.getHeaderValue("Pragma"));
        
        HttpServletSupport.addNoCacheHeaders(response);
        Assert.assertEquals(response.getHeaderValue("Cache-control"), "no-cache, no-store");
        Assert.assertEquals(response.getHeaderValue("Pragma"), "no-cache");
    }
    
    @Test public void testGetFullRequestURI(){
//        mock request doesn't do what we want, need to figure out something better
//        MockHttpServletRequest request = new MockHttpServletRequest();
//        
//        request.setScheme("http");
//        request.setServerName("example.org");
//        request.setRequestURI("/foo/bar");
//        request.setQueryString("baz=true");        
//        Assert.assertEquals(HttpServletSupport.getFullRequestUri(request), "http://example.org/foo/bar?baz=true");
//        
//        request.setScheme("https");
//        request.setServerPort(8443);
//        request.setQueryString(null);
//        Assert.assertEquals(HttpServletSupport.getFullRequestUri(request), "https://example.org:8443/foo/bar");
    }
    
    @Test public void testGetRequestPathWithoutContext(){
        
    }
    
    @Test public void testSetContentType(){
        
    }
    
    @Test public void testSetUTF8Encoding(){
        
    }
    
    @Test public void testValidateContentType() {
        MockHttpServletRequest request = new MockHttpServletRequest();
        
        // No Content-type
        Assert.assertTrue(HttpServletSupport.validateContentType(request, 
                CollectionSupport.singleton(MediaType.XML_UTF_8), 
                true, 
                false));
        
        Assert.assertFalse(HttpServletSupport.validateContentType(request, 
                CollectionSupport.singleton(MediaType.XML_UTF_8), 
                false, 
                false));
        
        // With charset parameter
        request.setContentType("text/xml; charset=utf-8");
        
        Assert.assertFalse(HttpServletSupport.validateContentType(request, 
                CollectionSupport.singleton(MediaType.create("application", "foobar")), 
                true, 
                false));
        
        Assert.assertTrue(HttpServletSupport.validateContentType(request, 
                CollectionSupport.setOf(MediaType.XML_UTF_8, MediaType.create("application", "foobar")), 
                true, 
                false));
        
        Assert.assertTrue(HttpServletSupport.validateContentType(request, 
                CollectionSupport.setOf(MediaType.XML_UTF_8, MediaType.create("application", "foobar")), 
                true, 
                true));
        
        Assert.assertTrue(HttpServletSupport.validateContentType(request, 
                CollectionSupport.setOf(MediaType.XML_UTF_8.withoutParameters(), MediaType.create("application", "foobar")), 
                true, 
                true));
        
        Assert.assertTrue(HttpServletSupport.validateContentType(request, 
                CollectionSupport.setOf(MediaType.ANY_TEXT_TYPE, MediaType.create("application", "foobar")), 
                true, 
                true));
        
        Assert.assertTrue(HttpServletSupport.validateContentType(request, 
                CollectionSupport.setOf(MediaType.ANY_TYPE, MediaType.create("application", "foobar")), 
                true, 
                true));
        
        // No parameters
        request.setContentType("text/xml");
        
        Assert.assertFalse(HttpServletSupport.validateContentType(request, 
                CollectionSupport.singleton(MediaType.create("application", "foobar")), 
                true, 
                false));
        
        Assert.assertTrue(HttpServletSupport.validateContentType(request, 
                CollectionSupport.setOf(MediaType.XML_UTF_8, MediaType.create("application", "foobar")), 
                true, 
                false));
        
        // Not valid, because the text/xml valid type includes parameters
        Assert.assertFalse(HttpServletSupport.validateContentType(request, 
                CollectionSupport.setOf(MediaType.XML_UTF_8, MediaType.create("application", "foobar")), 
                true, 
                true));
        
        Assert.assertTrue(HttpServletSupport.validateContentType(request, 
                CollectionSupport.setOf(MediaType.XML_UTF_8.withoutParameters(), MediaType.create("application", "foobar")), 
                true, 
                true));
        
        Assert.assertTrue(HttpServletSupport.validateContentType(request, 
                CollectionSupport.setOf(MediaType.ANY_TEXT_TYPE, MediaType.create("application", "foobar")), 
                true, 
                true));
        
        Assert.assertTrue(HttpServletSupport.validateContentType(request, 
                CollectionSupport.setOf(MediaType.ANY_TYPE, MediaType.create("application", "foobar")), 
                true, 
                true));
        
    }

    @Test
    public void testLanguageRanges() {
        MockHttpServletRequest request =  new MockHttpServletRequest();

        Assert.assertTrue(HttpServletSupport.getLanguageRange(request).isEmpty());
        request.addHeader("Accept-Language", "en-GB,fr-FR;q=0.7,en;q=0.3");
        List<LanguageRange> ranges = HttpServletSupport.getLanguageRange(request);
        Assert.assertEquals(ranges.size(), 4);
        request =  new MockHttpServletRequest();
        request.addHeader("Accept-Language", "en-ca;en-us");
        ranges = HttpServletSupport.getLanguageRange(request);
        Assert.assertTrue(ranges.isEmpty());
    }
}