/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.spring.security.factory;

import java.io.InputStream;
import java.security.cert.X509Certificate;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.cryptacular.util.CertUtil;
import org.springframework.beans.factory.BeanCreationException;
import org.springframework.beans.factory.FactoryBean;
import org.springframework.core.io.Resource;

/**
 * Spring bean factory for producing a {@link X509Certificate} from a {@link Resource}.
 * 
 * This factory bean supports DER and PEM encoded certificate resources.
 */
public class X509CertificateFactoryBean implements FactoryBean<X509Certificate> {

    /** Certificate chain resource. */
    @Nullable private Resource resource;

    /** The singleton instance of the certificate produced by this factory. */
    @Nullable private X509Certificate certificate;

    /**
     * Sets the certificate resource.
     * 
     * @param res certificate resource
     */
    public void setResource(@Nullable final Resource res) {
        resource = res;
    }

    /** {@inheritDoc} */
    @Override public X509Certificate getObject() throws Exception {
        if (certificate == null) {
            if (resource == null) {
                throw new BeanCreationException("Certificate resource must be provided in order to use this factory.");
            }

            assert resource != null;
            try (final InputStream is = resource.getInputStream()) {
                certificate = CertUtil.readCertificate(is);
            }
        }

        return certificate;
    }

    /** {@inheritDoc} */
    @Override @Nonnull public Class<?> getObjectType() {
        return X509Certificate.class;
    }

    /** {@inheritDoc} */
    @Override public boolean isSingleton() {
        return true;
    }

}