/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.security;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URL;

import javax.annotation.Nonnull;

import org.springframework.core.io.Resource;

import net.shibboleth.shared.logic.Constraint;

/**
 * Bridging class between {@link Resource} and {@link net.shibboleth.shared.resource.Resource}.
 */
public final class TestResourceConverter implements net.shibboleth.shared.resource.Resource {

    /** The cached Spring {@link Resource}. */
    private Resource springResource;

    /**
     * A private for shimming the provided input.
     * 
     * @param theResource the spring resource;
     */
    private TestResourceConverter(@Nonnull Resource theResource) {

        springResource = Constraint.isNotNull(theResource, "provided Spring Resource should not be null");
    }

    /**
     * Return a {@link Resource} that does all the work of the provided {@link Resource}.
     * 
     * <p>
     * If the input implements {@link Resource} then it is cast to the output, other a shim class is
     * generated.
     * </p>
     * 
     * @param springResource the input
     * @return a {@link Resource} which reflects what the Spring one does
     */
    @Nonnull public static net.shibboleth.shared.resource.Resource of(@Nonnull Resource springResource) {
        if (springResource instanceof net.shibboleth.shared.resource.Resource) {
            return (net.shibboleth.shared.resource.Resource) springResource;
        }
        return new TestResourceConverter(springResource);
    }

    /** {@inheritDoc} */
    @Override @Nonnull public InputStream getInputStream() throws IOException {
        return springResource.getInputStream();
    }

    /** {@inheritDoc} */
    @Override public boolean exists() {
        return springResource.exists();
    }

    /** {@inheritDoc} */
    @Override public boolean isReadable() {
        return springResource.isReadable();
    }

    /** {@inheritDoc} */
    @Override public boolean isOpen() {
        return springResource.isOpen();
    }

    /** {@inheritDoc} */
    @Override @Nonnull public URL getURL() throws IOException {
        return springResource.getURL();
    }

    /** {@inheritDoc} */
    @Override @Nonnull public URI getURI() throws IOException {
        return springResource.getURI();
    }

    /** {@inheritDoc} */
    @Override @Nonnull public File getFile() throws IOException {
        return springResource.getFile();
    }

    /** {@inheritDoc} */
    @Override public long contentLength() throws IOException {
        return springResource.contentLength();
    }

    /** {@inheritDoc} */
    @Override public long lastModified() throws IOException {
        return springResource.lastModified();
    }

    /** {@inheritDoc} */
    @Override @Nonnull public net.shibboleth.shared.resource.Resource createRelativeResource(
            @Nonnull String relativePath) throws IOException {

        return of(springResource.createRelative(relativePath));
    }

    /** {@inheritDoc} */
    @Override public String getFilename() {
        return springResource.getFilename();
    }

    /** {@inheritDoc} */
    @Override @Nonnull public String getDescription() {
        return springResource.getDescription();
    }

}
