/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.security.impl;

import java.io.IOException;
import java.io.InputStream;
import java.security.KeyException;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.time.Duration;
import java.util.HashMap;
import java.util.Map;

import javax.crypto.KeyGenerator;
import javax.script.ScriptException;

import net.shibboleth.shared.component.ComponentInitializationException;
import net.shibboleth.shared.scripting.EvaluableScript;

import org.testng.Assert;
import org.testng.annotations.BeforeMethod;
import org.testng.annotations.Test;

/**
 * Test for {@link ScriptedKeyStrategy}.
 */
public class ScriptedKeyStrategyTest {
    
    private Map<String,Object> customMap;
    
    private KeyGenerator keyGenerator;
    
    private ScriptedKeyStrategy strategy;
    
    @BeforeMethod public void setUp() throws ComponentInitializationException, NoSuchAlgorithmException, ScriptException, IOException {

        final SecureRandom random = new SecureRandom(); 
        keyGenerator = KeyGenerator.getInstance("AES");
        keyGenerator.init(random); 
        customMap = new HashMap<>();
        customMap.put("secret1", keyGenerator.generateKey());
        customMap.put("default", "secret1");
        
        final EvaluableScript script = new EvaluableScript();
        try (final InputStream in = getClass().getResourceAsStream("keyStrategyScript.js")) {
            script.setScript(in);
        }
        script.initialize();
        
        strategy = new ScriptedKeyStrategy();
        strategy.setUpdateInterval(Duration.ofSeconds(1));
        strategy.setKeyScript(script);
        strategy.setCustomObject(customMap);
        strategy.setCacheSize(1);
        strategy.initialize();
    }
    
    @Test(expectedExceptions=ComponentInitializationException.class)
    public void testNoScript() throws ComponentInitializationException {
        new ScriptedKeyStrategy().initialize();
    }
    
    @Test(expectedExceptions=ComponentInitializationException.class)
    public void testScriptFailure() throws ComponentInitializationException, ScriptException {
        final EvaluableScript script = new EvaluableScript();
        script.setScript("null");
        script.initialize();
        
        final ScriptedKeyStrategy estrategy = new ScriptedKeyStrategy();
        estrategy.setKeyScript(script);
        estrategy.initialize();
    }
    
    @Test public void testScriptedKeystoreKeyStrategy() throws Exception {
    
        
        Assert.assertEquals(strategy.getDefaultKey().getFirst(), "secret1");
        try {
            strategy.getKey("secret2");
            Assert.fail("secret2 should not exist");
        } catch (final KeyException e) {
    
        }
    
        customMap.put("secret2", keyGenerator.generateKey());
        customMap.put("default", "secret2");
        Thread.sleep(5000);
        Assert.assertEquals(strategy.getDefaultKey().getFirst(), "secret2");
        Assert.assertNotNull(strategy.getKey("secret1"));
    
        customMap.put("secret3", keyGenerator.generateKey());
        customMap.put("default", "secret3");
        customMap.remove("secret1");
        Thread.sleep(5000);
        Assert.assertEquals(strategy.getDefaultKey().getFirst(), "secret3");
        Assert.assertNotNull(strategy.getKey("secret2"));
        try {
            strategy.getKey("secret1");
            Assert.fail("secret1 should not exist");
        } catch (final KeyException e) {
    
        }
    }
    
}