/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package net.shibboleth.shared.security.impl;

import java.security.InvalidAlgorithmParameterException;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.util.HashSet;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.codec.binary.Hex;
import org.testng.Assert;
import org.testng.annotations.Test;

import net.shibboleth.shared.security.IdentifierGenerationStrategy;
import net.shibboleth.shared.security.IdentifierGenerationStrategy.ProviderType;
import net.shibboleth.shared.security.RandomIdentifierParameterSpec;

/** Unit test for {@link SecureRandomIdentifierGenerationStrategy}. */
public class SecureRandomIdentifierGenerationStrategyTest {

    /**
     * Test generateIdentifier by generating a large number of identifiers and seeing whether each one is a valid XML
     * ID, and that they are all different.
     */
    @Test public void testGenerateIdentifier() {
        final Pattern ncNamePattern = Pattern.compile("^[a-zA-Z_][a-zA-Z0-9_\\-\\.]+$");
        final IdentifierGenerationStrategy strat = IdentifierGenerationStrategy.getInstance(ProviderType.RANDOM);
        final int howMany = 1000;
        final Set<String> values = new HashSet<>(1000);
        for (int iteration = 1; iteration <= howMany; iteration++) {
            final String value = strat.generateIdentifier();

            // we shouldn't see the same value twice
            if (values.contains(value)) {
                Assert.fail("duplicate value " + value + " on iteration " + iteration);
            }
            values.add(value);

            // values should be valid NCNames
            final Matcher match = ncNamePattern.matcher(value);
            if (!match.matches()) {
                Assert.fail("value " + value + " is not a valid NCName on iteration " + iteration);
            }
        }
    }

    /**
     * Test generateIdentifier by generating a large number of identifiers and seeing whether each one is a valid XML
     * ID, and that they are all different.
     * 
     * @throws NoSuchAlgorithmException if the SHA1PRNG algorithm is not available
     * @throws InvalidAlgorithmParameterException 
     */
    @Test public void testConstructorWithSecureRandom() throws NoSuchAlgorithmException, InvalidAlgorithmParameterException {
        final Pattern ncNamePattern = Pattern.compile("^[a-zA-Z_][a-zA-Z0-9_\\-\\.]+$");
        final IdentifierGenerationStrategy strat =
                IdentifierGenerationStrategy.getInstance(ProviderType.RANDOM,
                        new RandomIdentifierParameterSpec(SecureRandom.getInstance("SHA1PRNG"), 16, new Hex()));
        final int howMany = 1000;
        final Set<String> values = new HashSet<>(1000);
        for (int iteration = 1; iteration <= howMany; iteration++) {
            final String value = strat.generateIdentifier();

            // we shouldn't see the same value twice
            if (values.contains(value)) {
                Assert.fail("duplicate value " + value + " on iteration " + iteration);
            }
            values.add(value);

            // values should be valid NCNames
            final Matcher match = ncNamePattern.matcher(value);
            if (!match.matches()) {
                Assert.fail("value " + value + " is not a valid NCName on iteration " + iteration);
            }
        }
    }

}