/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.shibboleth.shared.service;

import java.time.Instant;
import java.util.HashMap;
import java.util.Map;

import javax.annotation.Nonnull;

import com.codahale.metrics.Gauge;
import com.codahale.metrics.Metric;
import com.codahale.metrics.MetricFilter;
import com.codahale.metrics.MetricRegistry;
import com.codahale.metrics.MetricSet;

import net.shibboleth.shared.annotation.ParameterName;
import net.shibboleth.shared.annotation.constraint.Live;
import net.shibboleth.shared.annotation.constraint.NonnullAfterInit;
import net.shibboleth.shared.annotation.constraint.NotEmpty;
import net.shibboleth.shared.collection.CollectionSupport;
import net.shibboleth.shared.component.AbstractInitializableComponent;
import net.shibboleth.shared.component.ComponentInitializationException;
import net.shibboleth.shared.logic.Constraint;

/**
 * A set of gauges for a reloadable service.
 * @param <T> Type of service we are monitoring
 */
public class ReloadableServiceGaugeSet<T> extends AbstractInitializableComponent implements MetricSet, MetricFilter {

    /** The map of gauges. */
    @Nonnull private Map<String,Metric> gauges;
    
    /** The service to report on. */
    @NonnullAfterInit private ReloadableService<T> service;

    /** The metric Prefix. */
    @Nonnull @NotEmpty private final String metricPrefix;

    /**
     * Constructor.
     * 
     * @param metricName name to include in metric names produced by this set
     */
    public ReloadableServiceGaugeSet(@Nonnull @NotEmpty @ParameterName(name="metricName") final String metricName) {
        metricPrefix = Constraint.isNotEmpty(metricName, "Metric name cannot be null or empty");
        
        gauges = new HashMap<>();

        gauges.put(
                MetricRegistry.name(metricPrefix, "reload", "success"),
                new Gauge<Instant>() {
                    public Instant getValue() {
                        return service.getLastSuccessfulReloadInstant();
                    }
                });
        
        gauges.put(
                MetricRegistry.name(metricPrefix, "reload", "attempt"),
                new Gauge<Instant>() {
                    public Instant getValue() {
                        return service.getLastReloadAttemptInstant();
                    }
                });

        gauges.put(
                MetricRegistry.name(metricPrefix, "reload", "error"),
                new Gauge<String>() {
                    public String getValue() {
                        final Throwable cause = service.getReloadFailureCause();
                        return cause != null ? cause.getMessage() : null;
                    }
                });

    }
    
    /**
     * Get the service to report on.
     * 
     * @return service to report on
     */
    @NonnullAfterInit public ReloadableService<T> getService() {
        return service;
    }
    
    /**
     * Set the service to report on.
     * 
     * @param svc service instance
     */
    public void setService(@Nonnull final ReloadableService<T> svc) {
        checkSetterPreconditions();
        
        service = Constraint.isNotNull(svc, "ReloadableService cannot be null");
    }
    
    /** {@inheritDoc} */
    @Override
    protected void doInitialize() throws ComponentInitializationException {
        super.doInitialize();

        if (service == null) {
            throw new ComponentInitializationException("ReloadableService cannot be null");
        }
    }

    /** {@inheritDoc} */
    public Map<String,Metric> getMetrics() {
        return CollectionSupport.copyToMap(gauges);
    }

    /** {@inheritDoc} */
    public boolean matches(final String name, final Metric metric) {
        return gauges.containsKey(name);
    }
    
    /**
     * Get the underlying map of metrics.
     * 
     * @return map of metrics
     */
    @Nonnull @Live protected Map<String,Metric> getMetricMap() {
        return gauges;
    }

    /** Get the log prefix.
     * @return the log prefix (usually the metric name).
     */
    @Nonnull @NotEmpty protected final String getLogPrefix() {
        return metricPrefix;
    }
}