/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.service;

import java.io.IOException;

import javax.annotation.Nonnull;

/**
 * Any component that wants to be reloaded via the Service interface and Spring implements this interface.
 * 
 * The idea is that the attribute resolver will be
 * <code>
 * public class AttributeResolver extends AbstractServiceableComponent&lt;AttributeResolver&gt; implements
 *  AttributeResolver, ServiceableComponent&lt;ServiceableComponent&gt;.
 *  </code>
 *  AbstractServiceableComponent will do all the work around reload and synchronization.
 *  
 * @param <T> The underlying type of the component.
 */
public interface ServiceableComponent<T> extends AutoCloseable {

    /**
     * Extract the component that does the actual work.  Callers <em>MUST</em> have the ServiceableComponent
     * pinned at this stage.
     *
     * @return the component.
     */
    @Nonnull T getComponent();
    
    /** {@inheritDoc}
     * Note that this is explicitly declared to <em>not</em> throw an {@link IOException}
     */
     void close();
}
