/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.service.reloadable;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import net.shibboleth.shared.annotation.ParameterName;
import net.shibboleth.shared.annotation.constraint.NotEmpty;
import net.shibboleth.shared.logic.Constraint;
import net.shibboleth.shared.primitive.LoggerFactory;
import net.shibboleth.shared.service.ReloadableService;
import net.shibboleth.shared.service.ServiceException;
import net.shibboleth.shared.service.ServiceableComponent;

import org.slf4j.Logger;

import org.springframework.beans.factory.ObjectFactory;
import org.springframework.beans.factory.config.Scope;
import org.springframework.context.ApplicationContext;

/**
 * Custom Spring bean {@link Scope} that directs bean requests into a managed {@link ApplicationContext}.
 * 
 * @since 5.4.0
 */
public class ReloadableScope implements Scope {

    /** Scope indicating reloadability. */
    @Nonnull @NotEmpty public static final String SCOPE_RELOADABLE = "reloadable";
    
    /** Logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(ReloadableScope.class);
    
    /** Managed context service wrapper. */
    @Nonnull private final ReloadableService<ApplicationContext> reloadableService;
    
    /**
     * Constructor.
     *
     * @param service instance of Spring context to wrap
     */
    public ReloadableScope(
            @Nonnull @ParameterName(name="service") final ReloadableService<ApplicationContext> service) {
        reloadableService = Constraint.isNotNull(service, "ReloadableService cannot be null");
    }

    /** {@inheritDoc} */
    @Nonnull public Object get(@Nonnull final String name, @Nonnull final ObjectFactory<?> objectFactory) {
        log.debug("Accessing reloadable bean instance '{}'", name);
        try (final ServiceableComponent<ApplicationContext> component = reloadableService.getServiceableComponent()) {
            return component.getComponent().getBean(name);
        } catch (final ServiceException e) {
            throw new IllegalStateException("Reloadable bean context is unavailable", e);
        }
    }

    /** {@inheritDoc} */
    public Object remove(@Nonnull final String name) {
        throw new UnsupportedOperationException("No support for object removal");
    }

    /** {@inheritDoc} */
    public void registerDestructionCallback(@Nonnull final String name, @Nonnull final Runnable callback) {
        log.warn("Ignoring unsupported destruction callback for '{}'", name);
    }

    /** {@inheritDoc} */
    @Nullable public Object resolveContextualObject(@Nonnull final String key) {
        return null;
    }

    /** {@inheritDoc} */
    @Nullable public String getConversationId() {
        return null;
    }

}