/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.service.security.impl;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.slf4j.Logger;

import jakarta.servlet.ServletRequest;
import net.shibboleth.shared.annotation.ParameterName;
import net.shibboleth.shared.component.AbstractIdentifiableInitializableComponent;
import net.shibboleth.shared.logic.Constraint;
import net.shibboleth.shared.primitive.LoggerFactory;
import net.shibboleth.shared.security.AccessControl;
import net.shibboleth.shared.security.AccessControlService;
import net.shibboleth.shared.service.ReloadableService;
import net.shibboleth.shared.service.ServiceException;
import net.shibboleth.shared.service.ServiceableComponent;

/**
 * This class uses the {@link ReloadableService} concept to implement {@link AccessControlService}
 * to hide the details of pinning and unpinning the underlying service.
 */
public class DelegatingAccessControlService extends AbstractIdentifiableInitializableComponent
    implements AccessControlService {

    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(DelegatingAccessControlService.class);

    /** The service which manages the reloading. */
    @Nonnull private final ReloadableService<AccessControlService> service;

    /**
     * Constructor.
     * 
     * @param acService the service which will manage the loading.
     */
    public DelegatingAccessControlService(
            @Nonnull @ParameterName(name="acService") final ReloadableService<AccessControlService> acService) {
        service = Constraint.isNotNull(acService, "AccessControlService cannot be null");
    }

    /** {@inheritDoc} */
    @Override
    @Nonnull public AccessControl getInstance(@Nonnull final String name) {
        checkComponentActive();

        try (final ServiceableComponent<AccessControlService> component = service.getServiceableComponent()){
            return component.getComponent().getInstance(name);
        } catch (final ServiceException e) {
            log.error("AccessControlService '{}': Invalid configuration", getId(), e);
            return new AccessControl() {
                public boolean checkAccess(@Nonnull final ServletRequest request, @Nullable final String operation,
                        @Nullable final String resource) {
                    return false;
                }
            };
        }
    }
    
}