/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.spring.service;

import java.io.IOException;

import net.shibboleth.shared.collection.CollectionSupport;
import net.shibboleth.shared.service.ReloadableService;
import net.shibboleth.shared.service.ServiceableComponent;
import net.shibboleth.shared.spring.util.ApplicationContextBuilder;

import org.springframework.context.ApplicationContext;
import org.springframework.context.support.GenericApplicationContext;
import org.springframework.core.io.ClassPathResource;
import org.springframework.core.io.Resource;
import org.testng.Assert;
import org.testng.annotations.Test;

@SuppressWarnings("javadoc")
public class ReloadableBeanServiceTest {

    @Test public void reloadableService() throws IOException, InterruptedException {
        
        try (final GenericApplicationContext appCtx = new ApplicationContextBuilder()
                .setName("appCtx")
                .setServiceConfigurations(CollectionSupport.<Resource>singletonList(
                        new ClassPathResource("net/shibboleth/shared/spring/service/ReloadableBeans1.xml")))
                .build()) {
            final NonReloadableTestBean bean = appCtx.getBean("nonReloadableBean", NonReloadableTestBean.class);
            Assert.assertEquals(10, bean.getValue());
            
            final ReloadableTestBean child1 = bean.getChild();
            
            final ReloadableService<ApplicationContext> embedded =
                    appCtx.getBean("reloadableBeanService", ReloadableService.class);
            
            try (final ServiceableComponent<ApplicationContext> component = embedded.getServiceableComponent()) {
                assert(component != null);
                Assert.assertFalse(component.getComponent().containsLocalBean("reloadableBeanService"));
            }
            
            embedded.reload();

            final ReloadableTestBean child2 = bean.getChild();

            Assert.assertNotSame(child1, child2);
        }
    }

}