/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.spring.service;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.time.Duration;
import java.time.Instant;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import net.shibboleth.shared.collection.CollectionSupport;
import net.shibboleth.shared.service.ServiceException;
import net.shibboleth.shared.service.ServiceableComponent;
import net.shibboleth.shared.spring.util.ApplicationContextBuilder;

import org.springframework.beans.factory.BeanInitializationException;
import org.springframework.context.support.GenericApplicationContext;
import org.springframework.core.io.ClassPathResource;
import org.springframework.core.io.FileSystemResource;
import org.springframework.core.io.Resource;
import org.testng.Assert;
import org.testng.annotations.AfterMethod;
import org.testng.annotations.Test;

import com.google.common.io.ByteStreams;

/** {@link ReloadableSpringService} unit test. */
@SuppressWarnings("javadoc")
public class ReloadableSpringServiceTest {

    @Nonnull private static final Duration RELOAD_DELAY = Duration.ofMillis(100);

    @Nullable private File testFile;

    private void createPopulatedFile(@Nonnull final String dataPath) throws IOException {
        testFile = File.createTempFile("ReloadableSpringServiceTest", ".xml");
        overwriteFileWith(dataPath);
        if (testFile != null) {
            testFile.setLastModified(365*24*60*60*1000);
        }
    }
    
    @AfterMethod public void deleteFile() {
        final File file = testFile;
        if (null != file) {
            if (file.exists()) {
                file.delete();
            }
            testFile = null;
        }
    }

    @Nonnull private Resource testFileResource() {
        assert(testFile != null);
        return new FileSystemResource(testFile);
    }

    private void overwriteFileWith(@Nonnull final String newDataPath) throws IOException {
        try (final OutputStream stream = new FileOutputStream(testFile)) {
            ByteStreams.copy(new ClassPathResource(newDataPath).getInputStream(), stream);
        }
    }

    @Test(enabled=true) public void reloadableService() throws IOException, InterruptedException {
        final ReloadableSpringService<TestServiceableComponent> service =
                new ReloadableSpringService<>(TestServiceableComponent.class);

        createPopulatedFile("net/shibboleth/shared/spring/service/ServiceableBean1.xml");

        service.setFailFast(true);
        service.setId("reloadableService");
        service.setReloadCheckDelay(RELOAD_DELAY);
        service.setServiceConfigurations(CollectionSupport.singletonList(testFileResource()));

        service.start();

        AbstractServiceableComponent<TestServiceableComponent> serviceableComponent = service.getServiceableComponent();
        assert(serviceableComponent != null);
        final TestServiceableComponent component = serviceableComponent.getComponent();

        Assert.assertEquals("One", component.getTheValue());
        Assert.assertFalse(component.isDestroyed());

        serviceableComponent.unpinComponent();
        overwriteFileWith("net/shibboleth/shared/spring/service/ServiceableBean2.xml");

        long count = 70;
        while (count > 0 && !component.isDestroyed()) {
            Thread.sleep(RELOAD_DELAY.toMillis());
            count--;
        }
        Assert.assertTrue(component.isDestroyed(), "After 7 second initial component has still not be destroyed");

        //
        // The reload will have destroyed the old component
        //
        Assert.assertTrue(serviceableComponent.getComponent().isDestroyed());

        serviceableComponent = service.getServiceableComponent();
        assert(serviceableComponent != null);

        Assert.assertEquals(serviceableComponent.getComponent().getTheValue(), "Two");
        serviceableComponent.unpinComponent();
        service.stop();
        
        deleteFile();
    }

    @Test(enabled=true) public void deferedReload() throws IOException, InterruptedException {
        final ReloadableSpringService<TestServiceableComponent> service =
                new ReloadableSpringService<>(TestServiceableComponent.class);

        createPopulatedFile("net/shibboleth/shared/spring/service/ServiceableBean1.xml");

        service.setFailFast(true);
        service.setId("deferedReload");
        service.setReloadCheckDelay(RELOAD_DELAY);
        service.setServiceConfigurations(CollectionSupport.singletonList(testFileResource()));

        service.start();

        final TestServiceableComponent component;
        try (final ServiceableComponent<TestServiceableComponent> serviceableComponent = service.getServiceableComponent()) {
            assert(serviceableComponent != null);
            component = serviceableComponent.getComponent();
        }
        final Instant x = service.getLastReloadAttemptInstant();
        Assert.assertEquals(x,  service.getLastSuccessfulReloadInstant());

        Assert.assertEquals(component.getTheValue(), "One");
        Assert.assertFalse(component.isDestroyed());

        Thread.sleep(RELOAD_DELAY.toMillis() * 3);
        Assert.assertEquals(x,  service.getLastReloadAttemptInstant());

        overwriteFileWith("net/shibboleth/shared/spring/service/ServiceableBean2.xml");

        //
        // The reload will not have destroyed the old component yet
        //
        Assert.assertFalse(component.isDestroyed());

        long count = 70;
        TestServiceableComponent component2 = null;
        while (count > 0) {
            try (ServiceableComponent<TestServiceableComponent> serviceableComponent = service.getServiceableComponent()) {
                assert(serviceableComponent != null);
                component2 = serviceableComponent.getComponent();
                if ("Two".equals(component2.getTheValue())) {
                    break;
                }
            }
            component2 = null;
            Thread.sleep(RELOAD_DELAY.toMillis());
            count--;
        }
        Assert.assertNotNull(component2, "After 7 second initial component has still not got new value");
    
        count = 70;
        while (count > 0 && !component.isDestroyed()) {
            Thread.sleep(RELOAD_DELAY.toMillis());
            count--;
        }
        Assert.assertTrue(component.isDestroyed(), "After 7 second initial component has still not be destroyed");
        service.stop();
        deleteFile();
    }

    @Test public void testFailFast() throws IOException, InterruptedException {
        final ReloadableSpringService<TestServiceableComponent> service =
                new ReloadableSpringService<>(TestServiceableComponent.class);

        createPopulatedFile("net/shibboleth/shared/spring/service/BrokenBean1.xml");

        service.setFailFast(true);
        service.setId("testFailFast");
        service.setReloadCheckDelay(Duration.ZERO);
        service.setServiceConfigurations(CollectionSupport.singletonList(testFileResource()));

        try {
            service.start();
            Assert.fail("Expected to fail");
        } catch (final BeanInitializationException e) {
            // OK
        }
        
        try {
            service.getServiceableComponent();
            Assert.fail("Expected to fail");
        } catch (final ServiceException e) {
            // OK
        }

        overwriteFileWith("net/shibboleth/shared/spring/service/ServiceableBean2.xml");

        Thread.sleep(RELOAD_DELAY.toMillis() * 2);
        
        try {
            service.getServiceableComponent();
            Assert.fail("Expected to fail");
        } catch (final ServiceException e) {
            // OK
        }

        service.stop();
        deleteFile();
    }
    @Test public void testNotFailFast() throws IOException, InterruptedException {
        final ReloadableSpringService<TestServiceableComponent> service =
                new ReloadableSpringService<>(TestServiceableComponent.class);

        createPopulatedFile("net/shibboleth/shared/spring/service/BrokenBean1.xml");

        service.setFailFast(false);
        service.setId("testNotFailFast");
        service.setReloadCheckDelay(RELOAD_DELAY);
        service.setServiceConfigurations(CollectionSupport.singletonList(testFileResource()));

        service.start();

        try {
            service.getServiceableComponent();
            Assert.fail("Expected to fail");
        } catch (final ServiceException e) {
            // OK
        }

        overwriteFileWith("net/shibboleth/shared/spring/service/ServiceableBean2.xml");

        long count = 700;
        AbstractServiceableComponent<TestServiceableComponent> serviceableComponent = null;
        while (count > 0 && null == serviceableComponent) {
            count--;
            try {
                serviceableComponent = service.getServiceableComponent();
            } catch (final ServiceException e) {
                Thread.sleep(RELOAD_DELAY.toMillis());
            }
        }
        Assert.assertNotNull(serviceableComponent, "After 7 second component has still not initialized");
        assert(serviceableComponent != null);
        final TestServiceableComponent component = serviceableComponent.getComponent();
        Assert.assertEquals(component.getTheValue(), "Two");

        Assert.assertFalse(component.isDestroyed());
        serviceableComponent.unpinComponent();
        service.stop();

        count = 70;
        while (count > 0 && !component.isDestroyed()) {
            Thread.sleep(RELOAD_DELAY.toMillis());
            count--;
        }
        Assert.assertTrue(component.isDestroyed(), "After 7 seconds component has still not be destroyed");

        deleteFile();
    }

    @Test public void testApplicationContextAware() {

        final Resource parentResource = new ClassPathResource("net/shibboleth/shared/spring/service/ReloadableSpringService.xml");

        try (final GenericApplicationContext appCtx = new ApplicationContextBuilder()
                .setName("appCtx")
                .setServiceConfigurations(CollectionSupport.singletonList(parentResource))
                .build()) {
            final ReloadableSpringService<?> service = appCtx.getBean("testReloadableSpringService", ReloadableSpringService.class);
    
            Assert.assertNotNull(service.getParentContext(), "Parent context should not be null");
        }
    }
    
    @Test public void testBeanNameAware() {

        final Resource parentResource = new ClassPathResource("net/shibboleth/shared/spring/service/ReloadableSpringService.xml");

        try (final GenericApplicationContext appCtx = new ApplicationContextBuilder()
                .setName("appCtx")
                .setServiceConfigurations(CollectionSupport.singletonList(parentResource))
                .build()) {
            final ReloadableSpringService<?> service1 =
                    appCtx.getBean("testReloadableSpringService", ReloadableSpringService.class);
            Assert.assertEquals(service1.getId(), "testReloadableSpringService");

            final ReloadableSpringService<?> service2 =
                    appCtx.getBean("testReloadableSpringServiceWithCustomID", ReloadableSpringService.class);
            Assert.assertEquals(service2.getId(), "CustomID");
        }
    }

}