/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.spring.service;

import javax.annotation.Nonnull;

import org.slf4j.Logger;

import net.shibboleth.shared.annotation.constraint.NotEmpty;
import net.shibboleth.shared.component.AbstractInitializableComponent;
import net.shibboleth.shared.component.ComponentInitializationException;
import net.shibboleth.shared.primitive.LoggerFactory;

/**
 * Test bean to canary out behavior of reloadable bean service.
 */
public class ReloadableTestBean extends AbstractInitializableComponent {

    @Nonnull private final Logger log = LoggerFactory.getLogger(ReloadableTestBean.class);
    
    @Nonnull @NotEmpty private final String id;
    
    private int value;
    
    /**
     * Constructor.
     *
     * @param name name of bean
     * @param val value of bean
     */
    public ReloadableTestBean(@Nonnull @NotEmpty final String name, final int val) {
        id = name;
        value = val;
        log.debug("ReloadableTestBean {} created", id);
    }

    /** {@inheritDoc} */
    protected void doInitialize() throws ComponentInitializationException {
        super.doInitialize();
        
        log.debug("ReloadableTestBean {} initialized", id);
    }

    /** {@inheritDoc} */
    protected void doDestroy() {
        log.debug("ReloadableTestBean {} destroyed", id);
        
        super.doDestroy();
    }
    
    /**
     * Get bean value.
     * 
     * @return bean value
     */
    public int getValue() {
        return value;
    }
    
}