/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.spring.error;

import java.util.Map;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.springframework.web.context.WebApplicationContext;
import org.springframework.web.context.support.WebApplicationContextUtils;
import org.springframework.web.servlet.ModelAndView;
import org.springframework.web.servlet.handler.SimpleMappingExceptionResolver;

import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import net.shibboleth.shared.codec.HTMLEncoder;
import net.shibboleth.shared.codec.StringDigester;
import net.shibboleth.shared.primitive.LoggerFactory;
import net.shibboleth.shared.security.IdentifierGenerationStrategy;

/**
 * Custom Spring exception to view mapper that populates the view model with data
 * obtained via an extension function.
 * 
 * <p>As a default, the view model will include a reference to the active {@link HttpServletRequest}
 * and the {@link HTMLEncoder} class.</p>
 */
public class ExtendedMappingExceptionResolver extends SimpleMappingExceptionResolver {

    /** Model attribute carrying {@link HttpServletRequest}. */
    @Nonnull private static final String MODEL_ATTR_REQUEST = "request";

    /** Model attribute carrying {@link HttpServletResponse}. */
    @Nonnull private static final String MODEL_ATTR_RESPONSE = "response";

    /** Model attribute carrying {@link WebApplicationContext}. */
    @Nonnull private static final String MODEL_ATTR_SPRINGCONTEXT = "springContext";

    /** Model attribute carrying {@link WebApplicationContext#getEnvironment()}. */
    @Nonnull private static final String MODEL_ATTR_ENVIRONMENT = "environment";

    /** Model attribute carrying the {@link HTMLEncoder} class. */
    @Nonnull private static final String MODEL_ATTR_ENCODER = "encoder";

    /** Model attribute carrying the CSP digester. */
    @Nonnull private static final String MODEL_ATTR_DIGESTER = "cspDigester";

    /** Model attribute carrying the CSP nonce generator. */
    @Nonnull private static final String MODEL_ATTR_NONCE = "cspNonce";

    /** Model attribute carrying the custom object. */
    @Nonnull private static final String MODEL_ATTR_CUSTOM = "custom";

    /** Function to obtain extensions to view model. */
    @Nullable private final Function<HttpServletRequest,Map<String,Object>> viewModelExtenderFunction;

    /** CSP digester. */
    @Nullable private StringDigester cspDigester;

    /** CSP nonce generator. */
    @Nullable private IdentifierGenerationStrategy cspNonceGenerator;

    /** Slot for custom view object to inject. */
    @Nullable private Object customObject;
    
    /** Constructor. */
    public ExtendedMappingExceptionResolver() {
        this(null);
    }

    /**
     * Constructor.
     *
     * @param extender function to obtain extensions to view model
     */
    public ExtendedMappingExceptionResolver(@Nullable final Function<HttpServletRequest,Map<String,Object>> extender) {
        viewModelExtenderFunction = extender;
    }
    
    /**
     * Sets a CSP digester.
     * 
     * @param digester CSP digester
     * 
     * @since 9.1.0
     */
    public void setCSPDigester(@Nullable final StringDigester digester) {
        cspDigester = digester;
    }

    /**
     * Sets a CSP nonce generator.
     * 
     * @param generator CSP nonce generator
     * 
     * @since 9.1.0
     */
    public void setCSPNonceGenerator(@Nullable final IdentifierGenerationStrategy generator) {
        cspNonceGenerator = generator;
    }

    /**
     * Sets a custom object to inject into the view.
     * 
     * @param custom custom object
     * 
     * @since 9.1.0
     */
    public void setCustomObject(@Nullable final Object custom) {
        customObject = custom;
    }
    
    /** {@inheritDoc} */
    @Override
    protected ModelAndView doResolveException(@Nonnull final HttpServletRequest request,
            @Nonnull final HttpServletResponse response, @Nullable final Object handler, @Nonnull final Exception ex) {
        
        final ModelAndView view = super.doResolveException(request, response, handler, ex);
        if (view != null) {
            view.addObject(MODEL_ATTR_RESPONSE, response);
        }
        return view;
    }

    /** {@inheritDoc} */
    @Override
    @Nonnull protected ModelAndView getModelAndView(@Nonnull final String viewName, @Nonnull final Exception ex,
            @Nonnull final HttpServletRequest request) {
        
        LoggerFactory.getLogger(ex.getClass()).error("", ex);
        
        final ModelAndView view = super.getModelAndView(viewName, ex, request);
        view.addObject(MODEL_ATTR_REQUEST, request);
        view.addObject(MODEL_ATTR_ENCODER, HTMLEncoder.class);

        view.addObject(MODEL_ATTR_DIGESTER, cspDigester);
        view.addObject(MODEL_ATTR_NONCE, cspNonceGenerator);
        view.addObject(MODEL_ATTR_CUSTOM, customObject);
        
        final WebApplicationContext context =
                WebApplicationContextUtils.getWebApplicationContext(request.getServletContext());
        if (context != null) {
            view.addObject(MODEL_ATTR_SPRINGCONTEXT, context);
            view.addObject(MODEL_ATTR_ENVIRONMENT, context.getEnvironment());
        }
        
        final Function<HttpServletRequest,Map<String,Object>> local = viewModelExtenderFunction;
        if (local != null) {
            final Map<String,Object> exts = local.apply(request);
            if (exts != null) {
                view.addAllObjects(exts);
            }
        }
        
        return view;
    }

}