/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.spring.expression;

import java.util.function.BiPredicate;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.slf4j.Logger;

import org.springframework.expression.EvaluationContext;

import net.shibboleth.shared.annotation.ParameterName;
import net.shibboleth.shared.annotation.constraint.NotEmpty;
import net.shibboleth.shared.collection.Pair;
import net.shibboleth.shared.primitive.LoggerFactory;

/**
 * Predicate whose condition is defined by an Spring EL expression.
 * 
 * @param <T> first input type
 * @param <U> second input type
 * 
 * @since 6.1.0
 */
public class SpringExpressionBiPredicate<T,U> extends AbstractSpringExpressionEvaluator 
            implements BiPredicate<T,U> {
    
    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(SpringExpressionBiPredicate.class);

    /** Input types. */
    @Nullable private Pair<Class<T>,Class<U>> inputTypes;

    /**
     * Constructor.
     *
     * @param expression the expression to evaluate
     */
    public SpringExpressionBiPredicate(@Nonnull @NotEmpty @ParameterName(name="expression") final String expression) {
        super(expression);
        setOutputType(Boolean.class);
        setReturnOnError(false);
    }

    /**
     * Get the input type to be enforced.
     *
     * @return input type
     */
    @Nullable public Pair<Class<T>,Class<U>> getInputTypes() {
        return inputTypes;
    }

    /**
     * Set the input types to be enforced.
     *
     * @param types the input types
     */
    public void setInputTypes(@Nullable final Pair<Class<T>,Class<U>> types) {
        if (types != null && types.getFirst() != null && types.getSecond() != null) {
            inputTypes = types;
        } else {
            inputTypes = null;
        }
    }
    
    /**
     * Set value to return if an error occurs.
     * 
     * @param flag value to return
     */
    public void setReturnOnError(final boolean flag) {
        setReturnOnError(Boolean.valueOf(flag));
    }
    
    /** {@inheritDoc} */
    public boolean test(@Nullable final T first, @Nullable final U second) {
        final Pair<Class<T>,Class<U>> types = getInputTypes();
        if (null != types) {
            final Class<T> intype1 = types.getFirst();
            final Class<U> intype2 = types.getSecond();
            
            if (null != first && null != intype1 && !intype1.isInstance(first)) {
                log.error("Input of type {} was not of type {}", first.getClass(), intype1);
                return (boolean) returnError();
            }
            if (null != second && null != intype2 && !intype2.isInstance(second)) {
                log.error("Input of type {} was not of type {}", second.getClass(), intype2);
                return (boolean) returnError();
            }
        }

        final Object result = evaluate(first, second);
        return (boolean) (result != null ? result : returnError());
    }

    /**
     * Helper function to sanity check return-on-error object.
     * 
     * @return a boolean-valued error fallback
     * 
     * @throws ClassCastException if the installed fallback is null or non-Boolean
     */
    private boolean returnError() throws ClassCastException {
        final Object ret = getReturnOnError();
        if (ret instanceof Boolean) {
            return (boolean) ret;
        }
        
        throw new ClassCastException("Unable to cast return value to a boolean");
    }

    /** {@inheritDoc} */
    @Override
    protected void prepareContext(@Nonnull final EvaluationContext context, @Nullable final Object... input) {
        context.setVariable("input1", input != null ? input[0] : null);
        context.setVariable("input2", input != null ? input[1] : null);
    }
    
}