/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.spring.expression;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.slf4j.Logger;

import org.springframework.expression.EvaluationContext;

import net.shibboleth.shared.annotation.ParameterName;
import net.shibboleth.shared.annotation.constraint.NotEmpty;
import net.shibboleth.shared.primitive.LoggerFactory;

import java.util.function.Predicate;

/**
 * Predicate whose condition is defined by an Spring EL expression.
 * 
 * @param <T> type of input
 * 
 * @since 5.4.0
 */
public class SpringExpressionPredicate<T> extends AbstractSpringExpressionEvaluator implements Predicate<T> {
    
    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(SpringExpressionPredicate.class);

    /** Input type. */
    @Nullable private Class<T> inputTypeClass;

    /**
     * Constructor.
     *
     * @param expression the expression to evaluate
     */
    public SpringExpressionPredicate(@Nonnull @NotEmpty @ParameterName(name="expression") final String expression) {
        super(expression);
        setOutputType(Boolean.class);
    }

    /**
     * Get the input type to be enforced.
     *
     * @return input type
     * 
     * @since 6.1.0
     */
    @Nullable public Class<T> getInputType() {
        return inputTypeClass;
    }

    /**
     * Set the input type to be enforced.
     *
     * @param type input type
     * 
     * @since 6.1.0
     */
    public void setInputType(@Nullable final Class<T> type) {
        inputTypeClass = type;
    }
    
    /**
     * Set value to return if an error occurs (default is false).
     * 
     * @param flag flag to set
     */
    public void setReturnOnError(final boolean flag) {
        super.setReturnOnError(flag);
    }

    /** {@inheritDoc} */
    public boolean test(@Nullable final T input) {
        
        final Class<T> itype = getInputType();
        if (null != itype && null != input && !itype.isInstance(input)) {
            log.error("Input of type {} was not of type {}", input.getClass(), itype);
            return returnError();
        }

        final Object result = evaluate(input);
        return (boolean) (result != null ? result : returnError());
    }

    /**
     * Helper function to sanity check return-on-error object.
     * 
     * @return a boolean-valued error fallback
     * 
     * @throws ClassCastException if the installed fallback is null or non-Boolean
     */
    private boolean returnError() throws ClassCastException {
        final Object ret = getReturnOnError();
        if (ret instanceof Boolean) {
            return (boolean) ret;
        }
        
        throw new ClassCastException("Unable to cast return value to a boolean");
    }
    
    /** {@inheritDoc} */
    @Override
    protected void prepareContext(@Nonnull final EvaluationContext context, @Nullable final Object... input) {
        context.setVariable("input", input != null ? input[0] : null);
    }
    
}