/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.spring.config;

import java.util.List;
import java.util.Set;

import javax.annotation.Nonnull;

import org.springframework.beans.factory.xml.XmlBeanDefinitionReader;
import org.springframework.context.ApplicationContext;
import org.springframework.context.support.GenericApplicationContext;
import org.testng.Assert;
import org.testng.annotations.Test;

import net.shibboleth.shared.spring.custom.SchemaTypeAwareXMLBeanDefinitionReader;

/**
 * Test for {@link RelocatedBeanFactoryPostProcessor}.
 */
public class RelocatedBeanFactoryPostProcessorTest {

    /**
     * Wraps a {@link String}.
     */
    public static class OuterBean {
        private String innerString;
        public void setInnerString(@Nonnull final String str) {
            innerString = str;
        }
        public String getInnerString() {
            return innerString;
        }
    }

    /**
     * Wraps an {@link OuterBean}.
     */
    public static class OuterBeanWrapper {
        private OuterBean bean;
        public void setBean(@Nonnull final OuterBean b) {
            bean = b;
        }
        public OuterBean getBean() {
            return bean;
        }
    }

    /**
     * Wraps a list of strings.
     */
    public static class ListWrapper {
        private List<String> list;
        public void setList(@Nonnull final List<String> l) {
            list = l;
        }
        public List<String> getList() {
            return list;
        }
    }

    /**
     * Wraps a Set of strings.
     */
    public static class SetWrapper {
        private Set<String> set;
        public void setSet(@Nonnull final Set<String> s) {
            set = s;
        }
        public Set<String> getSet() {
            return set;
        }
    }

    /**
     * Test replacement of bean's class.
     */
    @Test public void testRelocated() {
        final ApplicationContext context = getContext("net/shibboleth/shared/spring/config/relocated.xml");
        
        final String foo = (String) context.getBean("foo");
        Assert.assertEquals(foo, "bar");

        final String frobnitz = (String) context.getBean("frobnitz");
        Assert.assertEquals(frobnitz, "bar");

        final var outerBean = context.getBean("outerBean", OuterBean.class);
        Assert.assertEquals(outerBean.getInnerString(), "missing-nested");

        final var outerBean2 = context.getBean("outerBean2", OuterBean.class);
        Assert.assertEquals(outerBean2.getInnerString(), "missing-parent-nested");

        final var layeredBeans = context.getBean("layeredBeans", OuterBeanWrapper.class);
        Assert.assertEquals(layeredBeans.getBean().getInnerString(), "doubly-nested");

        final var layeredList = context.getBean("layeredList", ListWrapper.class);
        Assert.assertEquals(layeredList.getList().size(), 2);
        Assert.assertEquals((String)layeredList.getList().get(0), "listentry");
        Assert.assertEquals((String)layeredList.getList().get(1), "also a string");

        final var layeredSet = context.getBean("layeredSet", SetWrapper.class);
        Assert.assertEquals(layeredSet.getSet().size(), 2);
        Assert.assertTrue(layeredSet.getSet().contains("setentry"));
        Assert.assertTrue(layeredSet.getSet().contains("also a string"));
    }

    private ApplicationContext getContext(final @Nonnull String config) {
        final GenericApplicationContext context = new GenericApplicationContext();
        final XmlBeanDefinitionReader beanDefinitionReader = new SchemaTypeAwareXMLBeanDefinitionReader(context);

        beanDefinitionReader.setValidationMode(XmlBeanDefinitionReader.VALIDATION_XSD);
        beanDefinitionReader.loadBeanDefinitions(config);
        context.refresh();
     
        return context;
    }

}