/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.spring.config;

import static org.testng.Assert.assertEquals;
import static org.testng.Assert.assertFalse;
import static org.testng.Assert.assertNull;
import static org.testng.Assert.assertTrue;
import static org.testng.Assert.fail;

import java.util.List;
import java.util.function.Predicate;

import org.springframework.beans.factory.BeanCreationException;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.testng.AbstractTestNGSpringContextTests;
import org.testng.annotations.Test;

/**
 *
 */
@ContextConfiguration({"predicateConverter.xml"})
public class StringBooleanToPredicateConverterTest extends AbstractTestNGSpringContextTests {
    
    static final boolean ALWAYS_PREDICATE = true;

    @Test public void integerTest() {
        assert(applicationContext != null);
        FourProps bean = applicationContext.getBean("integer", FourProps.class);
        if (!ALWAYS_PREDICATE) {
            assertEquals(bean.getNameAsInt(), 1);
            assertNull(bean.getNameAsList());
            assertNull(bean.getNameAsString());
            assertNull(bean.getNameAsPredicate());
        } else {
            assertFalse(bean.nameAsPredicate.test(null));
        }
    }

    @Test public void stringTest() {
        assert(applicationContext != null);
        FourProps bean = applicationContext.getBean("string", FourProps.class);
        if (!ALWAYS_PREDICATE) {
            assertEquals(bean.getNameAsString(), "string");
            assertNull(bean.getNameAsList());
            assertNull(bean.getNameAsInt());
            assertNull(bean.getNameAsPredicate());
        } else {
            assertFalse(bean.nameAsPredicate.test(null));
        }
    }

    @Test public void predTrueTest() {
        assert(applicationContext != null);
        FourProps bean = applicationContext.getBean("predTrue", FourProps.class);
        assertTrue(bean.getNameAsPredicate().test(null));
        assertNull(bean.getNameAsList());
        assertNull(bean.getNameAsString());
        assertNull(bean.getNameAsInt());
    }

    @Test public void predFalseTest() {
        assert(applicationContext != null);
        FourProps bean = applicationContext.getBean("predFalse", FourProps.class);
        assertFalse(bean.getNameAsPredicate().test(null));
        assertNull(bean.getNameAsList());
        assertNull(bean.getNameAsString());
        assertNull(bean.getNameAsInt());
    }
    
    @Test public void listTest() {
        assert(applicationContext != null);
        FourProps bean = applicationContext.getBean("list", FourProps.class);
        if (!ALWAYS_PREDICATE) {
            assertEquals(bean.getNameAsList().size(), 2);
            assertNull(bean.getNameAsInt());
            assertNull(bean.getNameAsString());
            assertNull(bean.getNameAsPredicate());
        } else {
            assertFalse(bean.nameAsPredicate.test(null));
        }
    }

    @Test public void predOnlyTest() {
        assert(applicationContext != null);
        OneProp bean = applicationContext.getBean("predOnly", OneProp.class);
        assertTrue(bean.getNameAsPredicate().test(null));
    }

    @Test(enabled = false) public void stringOnlyTest() {
        assert(applicationContext != null);
        try {
            final OneProp bean = applicationContext.getBean("stringOnly", OneProp.class);
            if (!ALWAYS_PREDICATE) {
                fail();
            } else {
                assertFalse(bean.nameAsPredicate.test(null));
            }
        } catch (final BeanCreationException b) {
            assertFalse(ALWAYS_PREDICATE);
        }
    }
    
    public static class FourProps {
        private String nameAsString;
        
        private Integer nameAsInt;
        
        private Predicate<?> nameAsPredicate;
        
        private List<String> nameAsList;
        
        /** Setter.
         * @param input The nameAsInt to set.
         */
        public void setName(final Integer input) {
            nameAsInt = input;
        }
        
        /** Setter.
         * @param input The nameAsString to set.
         */
        public void setName(final String input) {
            nameAsString = input;
        }
        
        /** Setter.
         * @param input The nameAsPredicate to set.
         */
        public void setName(final Predicate<?> input) {
            nameAsPredicate = input;
        }
        
        /** Setter.
         * @param input The nameAsList to set.
         */
        public void setNameAsList(List<String> input) {
            nameAsList = input;
        }
        
        /** Getter.
         * @return Returns the nameAsInt.
         */
        public Integer getNameAsInt() {
            return nameAsInt;
        }
        
        /** Getter.
         * @return Returns the nameAsPredicate.
         */
        public Predicate<?> getNameAsPredicate() {
            return nameAsPredicate;
        }
        
        /** Default Getter.
         * This keeps Spring happy.  See JSSH-38 for why.
         * @return Returns the nameAsPredicate.
         */
        public Predicate<?> getName() {
            return nameAsPredicate;
        }

        /** Getter.
         * @return Returns the nameAsString.
         */
        public String getNameAsString() {
            return nameAsString;
        }
        
        /** Getter.
         * @return Returns the nameAsList.
         */
        public List<String> getNameAsList() {
            return nameAsList;
        }
    }
    
    public static class OneProp {
        private Predicate<?> nameAsPredicate;
        
        /**
         * @param input The nameAsPredicate to set.
         */
        public void setName(final Predicate<?> input) {
            nameAsPredicate = input;
        }
        
        /**
         * @return Returns the nameAsPredicate.
         */
        public Predicate<?> getNameAsPredicate() {
            return nameAsPredicate;
        }
        
    }

}
