/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.spring.resource;

import java.io.IOException;
import java.nio.file.Path;
import java.util.Collection;

import org.springframework.context.support.GenericApplicationContext;
import org.springframework.core.io.ClassPathResource;
import org.springframework.core.io.Resource;
import org.springframework.mock.env.MockPropertySource;
import org.testng.Assert;
import org.testng.annotations.BeforeClass;
import org.testng.annotations.Test;

import net.shibboleth.shared.collection.CollectionSupport;
import net.shibboleth.shared.spring.util.ApplicationContextBuilder;

@SuppressWarnings("javadoc")
public class Idp1326{
    
    private String parentAsString;
    private String separator;
    private Resource resource;
    
    @BeforeClass public void getFileSystemDetails() throws IOException {
        resource = new ClassPathResource("net/shibboleth/shared/spring/resource/idp1326.xml");
        final Path path = resource.getFile().toPath();
        
        separator = path.getFileSystem().getSeparator();
        Assert.assertEquals(separator.length(), 1);
        parentAsString = path.getParent().toAbsolutePath().toString();
    }
    

    /*
     * Fails on Windows
     */
    @Test(enabled=false) public void testPropFileNative() {
        testPropFile(parentAsString);
    }
    
    @Test public void testPropFileJava() {
        final StringBuilder sb = new StringBuilder(parentAsString);
        if (!"/".equals(separator)) {
            int i = sb.indexOf(separator);
            while (i > 0) {
                sb.replace(i, i+1, "/");
                i = sb.indexOf(separator);
            }
        }
        testPropFile(sb.toString());
    }

    
    private void testPropFile(String idpHome) {
        
        final ApplicationContextBuilder builder = new ApplicationContextBuilder();
        
        final MockPropertySource mockEnvVars = new MockPropertySource();
        mockEnvVars.setProperty("idp.home", idpHome);
        builder.setPropertySources(CollectionSupport.singletonList(mockEnvVars));
        builder.setServiceConfiguration(resource);
        
        final GenericApplicationContext context = builder.build();
        
        final Collection<String> beans = context.getBeansOfType(String.class).values();
        Assert.assertEquals(beans.size(), 1);

        Assert.assertEquals(beans.iterator().next(), idpHome);
        
        context.close();
    }

}