/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.collection;

import java.io.Serializable;
import java.time.Instant;
import java.time.ZonedDateTime;
import java.time.chrono.ChronoZonedDateTime;
import java.time.chrono.Chronology;
import java.time.temporal.Temporal;
import java.time.temporal.TemporalAdjuster;
import java.util.Arrays;
import java.util.List;

import org.testng.Assert;
import org.testng.annotations.Test;

/** Unit test for {@link LockableClassToInstanceMultiMap}. */
public class LockableClassToInstanceMultiMapTest {

    @Test public void testClearIsEmpty() {
        LockableClassToInstanceMultiMap<Object> map = new LockableClassToInstanceMultiMap<>();

        map.clear();
        Assert.assertTrue(map.isEmptyWithLock());

        map.put(new Object());
        Assert.assertFalse(map.isEmptyWithLock());

        map.clear();
        Assert.assertTrue(map.isEmptyWithLock());
    }

    @Test public void testKeysAndContainsKey() {
        LockableClassToInstanceMultiMap<Temporal> map = new LockableClassToInstanceMultiMap<>();
        populate(map);
        Assert.assertEquals(map.keysWithLock().size(), 2);
        Assert.assertFalse(map.containsKeyWithLock(null));
        Assert.assertFalse(map.containsKeyWithLock(Chronology.class));
        Assert.assertFalse(map.containsKeyWithLock(Temporal.class));
        Assert.assertFalse(map.containsKeyWithLock(TemporalAdjuster.class));
        Assert.assertTrue(map.containsKeyWithLock(ZonedDateTime.class));
        Assert.assertFalse(map.containsKeyWithLock(ChronoZonedDateTime.class));
        Assert.assertFalse(map.containsKeyWithLock(Comparable.class));
        Assert.assertFalse(map.containsKeyWithLock(Serializable.class));
        Assert.assertTrue(map.containsKeyWithLock(Instant.class));

        map = new LockableClassToInstanceMultiMap<>(true);
        populate(map);
        Assert.assertEquals(map.keysWithLock().size(), 8);
        Assert.assertFalse(map.containsKeyWithLock(null));
        Assert.assertFalse(map.containsKeyWithLock(Chronology.class));
        Assert.assertTrue(map.containsKeyWithLock(Temporal.class));
        Assert.assertTrue(map.containsKeyWithLock(TemporalAdjuster.class));
        Assert.assertTrue(map.containsKeyWithLock(ZonedDateTime.class));
        Assert.assertTrue(map.containsKeyWithLock(ChronoZonedDateTime.class));
        Assert.assertTrue(map.containsKeyWithLock(Comparable.class));
        Assert.assertTrue(map.containsKeyWithLock(Serializable.class));
        Assert.assertTrue(map.containsKeyWithLock(Instant.class));
    }

    @Test public void testValuesAndContainsValues() {
        LockableClassToInstanceMultiMap<Temporal> map = new LockableClassToInstanceMultiMap<>();

        ZonedDateTime now = ZonedDateTime.now();
        assert now!=null;
        map.put(now);

        ZonedDateTime now100 = now.plusMinutes(100);
        assert now100!=null;
        map.put(now100);

        Instant instant = Instant.now();
        assert instant!=null;
        map.put(instant);

        Assert.assertEquals(map.valuesWithLock().size(), 3);
        Assert.assertFalse(map.containsValueWithLock(null));
        Assert.assertFalse(map.containsValueWithLock(now.minusMinutes(100)));
        Assert.assertFalse(map.containsValueWithLock(instant.minusSeconds(100)));
        Assert.assertTrue(map.containsValueWithLock(instant));
        Assert.assertTrue(map.containsValueWithLock(now));
        Assert.assertTrue(map.containsValueWithLock(now100));
    }

    @Test public void testEquals() {
        final LockableClassToInstanceMultiMap<Temporal> map = new LockableClassToInstanceMultiMap<>();
        final LockableClassToInstanceMultiMap<Temporal> map2 = new LockableClassToInstanceMultiMap<>();
        final LockableClassToInstanceMultiMap<Temporal> map3 = new LockableClassToInstanceMultiMap<>();

        final ZonedDateTime now = ZonedDateTime.now();
        assert now!=null;
        map.putWithLock(now);
        map2.putWithLock(now);
        map3.putWithLock(now);

        final ZonedDateTime now100 = now.plusMinutes(100);
        assert now100!=null;
        map.putWithLock(now100);
        map2.putWithLock(now100);
        map3.putWithLock(now100);

        final Instant instant = Instant.now();
        assert instant!=null;
        map.putWithLock(instant);
        map2.putWithLock(instant);

        Assert.assertTrue(map.equals(map2));
        Assert.assertFalse(map.equals(map3));

        Assert.assertEquals(map.hashCode(), map2.hashCode());
        Assert.assertNotEquals(map.hashCode(), map3.hashCode());

    }

    @Test public void testGet() {
        LockableClassToInstanceMultiMap<Temporal> map = new LockableClassToInstanceMultiMap<>();
        populate(map);

        List<?> values = map.getWithLock(null);
        Assert.assertEquals(values.size(), 0);

        values = map.getWithLock(ZonedDateTime.class);
        Assert.assertEquals(values.size(), 2);

        values = map.getWithLock(Instant.class);
        Assert.assertEquals(values.size(), 1);
    }

    @Test public void testNoIndexedDuplicateValues() {
        LockableClassToInstanceMultiMap<Object> map = new LockableClassToInstanceMultiMap<>(true);

        map.putWithLock(new FooBarImpl());

        Assert.assertEquals(map.getWithLock(Foo.class).size(), 1);
        Assert.assertEquals(map.getWithLock(Bar.class).size(), 1);
        Assert.assertEquals(map.getWithLock(AbstractFoo.class).size(), 1);
        Assert.assertEquals(map.getWithLock(AbstractFooBar.class).size(), 1);
        Assert.assertEquals(map.getWithLock(FooBarImpl.class).size(), 1);
    }

    @Test public void testDuplicateInsertions() {
        LockableClassToInstanceMultiMap<Object> map = new LockableClassToInstanceMultiMap<>(true);

        FooBarImpl fb = new FooBarImpl();

        map.putWithLock(fb);
        map.putWithLock(fb);

        Assert.assertEquals(map.valuesWithLock().size(), 1);

        Assert.assertEquals(map.getWithLock(Foo.class).size(), 1);
        Assert.assertEquals(map.getWithLock(Bar.class).size(), 1);
        Assert.assertEquals(map.getWithLock(AbstractFoo.class).size(), 1);
        Assert.assertEquals(map.getWithLock(AbstractFooBar.class).size(), 1);
        Assert.assertEquals(map.getWithLock(FooBarImpl.class).size(), 1);
    }

    @Test public void testRemoveValue() {
        LockableClassToInstanceMultiMap<Object> map = new LockableClassToInstanceMultiMap<>(true);

        FooBarImpl fb = new FooBarImpl();
        FooImpl f = new FooImpl();

        map.putWithLock(fb); // This is what we'll remove.
        map.putWithLock(f); // This is canary to test that its indexes don't disappear.

        Assert.assertTrue(map.containsValueWithLock(fb));
        Assert.assertTrue(map.containsValueWithLock(f));

        Assert.assertTrue(map.containsKeyWithLock(Foo.class));
        Assert.assertTrue(map.containsKeyWithLock(AbstractFoo.class));
        Assert.assertTrue(map.containsKeyWithLock(FooImpl.class));

        Assert.assertEquals(map.getWithLock(Foo.class).size(), 2);
        Assert.assertEquals(map.getWithLock(AbstractFoo.class).size(), 2);
        Assert.assertEquals(map.getWithLock(FooImpl.class).size(), 1);

        Assert.assertTrue(map.containsKeyWithLock(Bar.class));
        Assert.assertTrue(map.containsKeyWithLock(AbstractFooBar.class));
        Assert.assertTrue(map.containsKeyWithLock(FooBarImpl.class));

        Assert.assertEquals(map.getWithLock(Bar.class).size(), 1);
        Assert.assertEquals(map.getWithLock(AbstractFooBar.class).size(), 1);
        Assert.assertEquals(map.getWithLock(FooBarImpl.class).size(), 1);

        map.remove(fb);

        Assert.assertFalse(map.containsValueWithLock(fb));
        Assert.assertTrue(map.containsValueWithLock(f));

        Assert.assertTrue(map.containsKeyWithLock(Foo.class));
        Assert.assertTrue(map.containsKeyWithLock(AbstractFoo.class));
        Assert.assertTrue(map.containsKeyWithLock(FooImpl.class));

        Assert.assertEquals(map.getWithLock(Foo.class).size(), 1);
        Assert.assertEquals(map.getWithLock(AbstractFoo.class).size(), 1);
        Assert.assertEquals(map.getWithLock(FooImpl.class).size(), 1);

        Assert.assertFalse(map.containsKeyWithLock(Bar.class));
        Assert.assertFalse(map.containsKeyWithLock(AbstractFooBar.class));
        Assert.assertFalse(map.containsKeyWithLock(FooBarImpl.class));

        Assert.assertEquals(map.getWithLock(Bar.class).size(), 0);
        Assert.assertEquals(map.getWithLock(AbstractFooBar.class).size(), 0);
        Assert.assertEquals(map.getWithLock(FooBarImpl.class).size(), 0);
    }

    @Test public void testRemoveByType() {
        LockableClassToInstanceMultiMap<Object> map = new LockableClassToInstanceMultiMap<>(true);

        FooBarImpl fb = new FooBarImpl();
        FooImpl f = new FooImpl();

        map.putWithLock(fb);
        map.putWithLock(f);

        Assert.assertTrue(map.containsValueWithLock(fb));
        Assert.assertTrue(map.containsValueWithLock(f));

        Assert.assertTrue(map.containsKeyWithLock(Foo.class));
        Assert.assertTrue(map.containsKeyWithLock(AbstractFoo.class));
        Assert.assertTrue(map.containsKeyWithLock(FooImpl.class));

        Assert.assertEquals(map.getWithLock(Foo.class).size(), 2);
        Assert.assertEquals(map.getWithLock(AbstractFoo.class).size(), 2);
        Assert.assertEquals(map.getWithLock(FooImpl.class).size(), 1);

        Assert.assertTrue(map.containsKeyWithLock(Bar.class));
        Assert.assertTrue(map.containsKeyWithLock(AbstractFooBar.class));
        Assert.assertTrue(map.containsKeyWithLock(FooBarImpl.class));

        Assert.assertEquals(map.getWithLock(Bar.class).size(), 1);
        Assert.assertEquals(map.getWithLock(AbstractFooBar.class).size(), 1);
        Assert.assertEquals(map.getWithLock(FooBarImpl.class).size(), 1);

        map.removeWithLock(Bar.class);

        Assert.assertFalse(map.containsValueWithLock(fb));
        Assert.assertTrue(map.containsValueWithLock(f));

        Assert.assertTrue(map.containsKeyWithLock(Foo.class));
        Assert.assertTrue(map.containsKeyWithLock(AbstractFoo.class));
        Assert.assertTrue(map.containsKeyWithLock(FooImpl.class));

        Assert.assertEquals(map.getWithLock(Foo.class).size(), 1);
        Assert.assertEquals(map.getWithLock(AbstractFoo.class).size(), 1);
        Assert.assertEquals(map.getWithLock(FooImpl.class).size(), 1);

        Assert.assertFalse(map.containsKeyWithLock(Bar.class));
        Assert.assertFalse(map.containsKeyWithLock(AbstractFooBar.class));
        Assert.assertFalse(map.containsKeyWithLock(FooBarImpl.class));

        Assert.assertEquals(map.getWithLock(Bar.class).size(), 0);
        Assert.assertEquals(map.getWithLock(AbstractFooBar.class).size(), 0);
        Assert.assertEquals(map.getWithLock(FooBarImpl.class).size(), 0);

    }

    @Test public void testRemoveAll() {
        LockableClassToInstanceMultiMap<Object> map = new LockableClassToInstanceMultiMap<>(true);

        FooImpl f1 = new FooImpl();
        FooImpl f2 = new FooImpl();
        FooImpl f3 = new FooImpl();

        FooBarImpl fb1 = new FooBarImpl();
        FooBarImpl fb2 = new FooBarImpl();
        FooBarImpl fb3 = new FooBarImpl();

        map.putWithLock(f1);
        map.putWithLock(f2);
        map.putWithLock(f3);
        map.putWithLock(fb1);
        map.putWithLock(fb2);
        map.putWithLock(fb3);

        Assert.assertEquals(map.valuesWithLock().size(), 6);
        Assert.assertEquals(map.getWithLock(Foo.class).size(), 6);
        Assert.assertEquals(map.getWithLock(Bar.class).size(), 3);

        map.removeAllWithLock(Arrays.asList(f1, f2, fb1));

        Assert.assertEquals(map.valuesWithLock().size(), 3);
        Assert.assertEquals(map.getWithLock(Foo.class).size(), 3);
        Assert.assertEquals(map.getWithLock(Bar.class).size(), 2);

        map.removeAllWithLock(Arrays.asList(fb2, fb3));

        Assert.assertEquals(map.valuesWithLock().size(), 1);
        Assert.assertEquals(map.getWithLock(Foo.class).size(), 1);
        Assert.assertEquals(map.getWithLock(Bar.class).size(), 0);
        Assert.assertFalse(map.containsKeyWithLock(Bar.class));

        map.removeAllWithLock(Arrays.asList(f3));

        Assert.assertEquals(map.valuesWithLock().size(), 0);
        Assert.assertTrue(map.isEmptyWithLock());
        Assert.assertEquals(map.getWithLock(Foo.class).size(), 0);
        Assert.assertEquals(map.getWithLock(Bar.class).size(), 0);
        Assert.assertFalse(map.containsKeyWithLock(Foo.class));
        Assert.assertFalse(map.containsKeyWithLock(Bar.class));
    }

    protected void populate(ClassToInstanceMultiMap<Temporal> map) {
        ZonedDateTime now = ZonedDateTime.now();
        assert now!=null;
        map.put(now);

        ZonedDateTime now100 = now.plusMinutes(100);
        assert now100!=null;
        map.put(now100);

        Instant instant = Instant.now();
        assert instant!=null;
        map.put(instant);
    }

    // Test classes and interfaces

    public interface Foo {
    };

    public interface Bar extends Foo {
    };

    public abstract class AbstractFoo implements Foo {
    };

    public class FooImpl extends AbstractFoo {
    };

    public abstract class AbstractFooBar extends AbstractFoo implements Bar {
    };

    public class FooBarImpl extends AbstractFooBar {
    };

}