/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.xml;

import static org.testng.Assert.*;

import java.io.IOException;
import java.io.InputStream;
import java.time.Duration;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.Arrays;
import java.util.List;

import javax.annotation.Nonnull;
import javax.xml.namespace.QName;
import javax.xml.parsers.DocumentBuilder;

import net.shibboleth.shared.annotation.constraint.NonnullBeforeTest;
import net.shibboleth.shared.annotation.constraint.NotEmpty;
import net.shibboleth.shared.component.ComponentInitializationException;
import net.shibboleth.shared.logic.Constraint;
import net.shibboleth.shared.logic.ConstraintViolationException;
import net.shibboleth.shared.xml.impl.BasicParserPool;

import org.testng.annotations.BeforeClass;
import org.testng.annotations.BeforeMethod;
import org.testng.annotations.Test;
import org.w3c.dom.Attr;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.SAXException;

/**
 * Tests for {@link AttributeSupport};
 */
@SuppressWarnings("javadoc")
public class AttributeSupportTest {

    // Contants to test against
    @Nonnull @NotEmpty private static final String TEST_NS = "http://example.org/NameSpace";

    @Nonnull @NotEmpty private static final String TEST_PREFIX = "testns";

    @Nonnull @NotEmpty private static final String TEST_ID_ATTRIBUTE = "testAttributeName";

    @Nonnull @NotEmpty private static final String TEST_ID_PREFIXEDATTRIBUTE = TEST_PREFIX + ":" + TEST_ID_ATTRIBUTE;

    @Nonnull @NotEmpty private static final String TEST_ID_ATTRIBUTE_VALUE = "IDAttrVALUE";

    // Set up at start of all methods
    @NonnullBeforeTest private QName idAttrQName;

    @NonnullBeforeTest private Element goodBaseIdSpaceLang;

    @NonnullBeforeTest private Element noBaseIdSpaceLang;

    @NonnullBeforeTest private Element badSpace;

    @NonnullBeforeTest private Element preserveSpace;

    @NonnullBeforeTest private Element attributes;

    @NonnullBeforeTest private Element createdElement;

    // Reset before each method.
    private Document document;

    @NonnullBeforeTest private BasicParserPool parserPool;

    @BeforeClass public void setUp() throws XMLParserException, ComponentInitializationException, SAXException,
            IOException {
        parserPool = new BasicParserPool();
        parserPool.initialize();

        DocumentBuilder builder = parserPool.getBuilder();
        try (final InputStream s = getClass().getResourceAsStream("/net/shibboleth/shared/xml/attributeSupportTest.xml")) {
            idAttrQName = new QName(TEST_NS, TEST_ID_ATTRIBUTE, TEST_PREFIX);

            Document testFile = builder.parse(s);

            Element root = (Element) testFile.getFirstChild();

            // Skip whitespace, grab first element
            goodBaseIdSpaceLang = (Element) root.getFirstChild().getNextSibling();
            assertEquals(goodBaseIdSpaceLang.getLocalName(), "GoodBaseIdSpaceLang");

            noBaseIdSpaceLang = (Element) goodBaseIdSpaceLang.getNextSibling().getNextSibling();
            assertEquals(noBaseIdSpaceLang.getLocalName(), "NoBaseIdSpaceLang");

            badSpace = (Element) noBaseIdSpaceLang.getNextSibling().getNextSibling();
            assertEquals(badSpace.getLocalName(), "BadSpace");

            preserveSpace = (Element) badSpace.getNextSibling().getNextSibling();
            assertEquals(preserveSpace.getLocalName(), "PreserveSpace");

            attributes = (Element) preserveSpace.getNextSibling().getNextSibling();
            assertEquals(attributes.getLocalName(), "AttributeTest");

        } finally {
            parserPool.returnBuilder(builder);
        }
    }

    @BeforeMethod public void resetCreatedElement() throws XMLParserException {

        DocumentBuilder builder = parserPool.getBuilder();
        try {
            document = builder.newDocument();
            createdElement = document.createElement("TestElement");
            Attr attr = document.createAttributeNS(TEST_NS, TEST_ID_PREFIXEDATTRIBUTE);
            attr.setValue(TEST_ID_ATTRIBUTE_VALUE);
            Element el = document.createElement("ChildElement");
            el.setAttributeNode(attr);
            el.setIdAttributeNode(attr, true);
            createdElement.appendChild(el);
        } finally {
            parserPool.returnBuilder(builder);

        }
    }

    /**
     * Strictly speaking this test is for the parser. But we may significant assumptions on this test passing.
     * 
     * @throws XMLParserException if badness happens.
     * @throws ComponentInitializationException if badness happens.
     * @throws IOException if badness happens.
     */
    @Test public void testBadNS() throws XMLParserException, ComponentInitializationException, IOException {

        DocumentBuilder builder = parserPool.getBuilder();

        try (final InputStream s = getClass().getResourceAsStream("/net/shibboleth/shared/xml/badNS1.xml")) {
            boolean thrown = false;
            try {
                builder.parse(s);
            } catch (SAXException e) {
                thrown = true;
            }
            assertTrue(
                    thrown,
                    "xmlns: declaration with name other than xml and namespace of http://www.w3.org/XML/1998/namespace should throw an error ");
        }

        try (final InputStream s = getClass().getResourceAsStream("/net/shibboleth/shared/xml/badNS2.xml")) {
            boolean thrown = false;
            try {
                builder.parse(s);
            } catch (SAXException e) {
                thrown = true;
            }
            assertTrue(thrown,
                    "xmlns:xml with namespace other than http://www.w3.org/XML/1998/namespace should throw an error ");
        }

        parserPool.returnBuilder(builder);
    }

    @Test public void testGetXMLId() {
        assertEquals(AttributeSupport.getXMLId(goodBaseIdSpaceLang), "identifierGoodBaseIdSpaceLang",
                "Identifier mismatch");
        assertNull(AttributeSupport.getXMLId(noBaseIdSpaceLang), "Identifier found erroneously");
        assertEquals(AttributeSupport.getXMLId(badSpace), "identifierBadSpace", "Identifier mismatch");
        assertEquals(AttributeSupport.getXMLId(preserveSpace), "identifierPreserveSpace", "Identifier mismatch");

        // test Add now that we know that get works
        boolean thrown = false;
        try {
            AttributeSupport.addXMLId(createdElement, nullValue());
        } catch (ConstraintViolationException e) {
            thrown = true;
        }
        assertTrue(thrown, "null string to addXMLId");

        thrown = false;
        try {
            AttributeSupport.addXMLId(nullValue(), "fr");
        } catch (ConstraintViolationException e) {
            thrown = true;
        }
        assertTrue(thrown, "null element to addXMLId");

        assertNull(AttributeSupport.getXMLId(createdElement), "xml:space found erroneously (test setup failure)");
        AttributeSupport.addXMLId(createdElement, TEST_ID_ATTRIBUTE_VALUE);
        assertEquals(AttributeSupport.getXMLId(createdElement), TEST_ID_ATTRIBUTE_VALUE, "addXMLId failed");

    }

    @Test public void testXMLBase() {
        assertEquals(AttributeSupport.getXMLBase(goodBaseIdSpaceLang), "http://example.org/base",
                "xml:base mismatch");
        assertNull(AttributeSupport.getXMLBase(noBaseIdSpaceLang), "xml:base found erroneously");

        // test Add
        boolean thrown = false;
        try {
            AttributeSupport.addXMLBase(createdElement, nullValue());
        } catch (ConstraintViolationException e) {
            thrown = true;
        }
        assertTrue(thrown, "null string to addXMLBase");

        thrown = false;
        try {
            AttributeSupport.addXMLBase(nullValue(), "foo");
        } catch (ConstraintViolationException e) {
            thrown = true;
        }
        assertTrue(thrown, "null element to addXMLBase");

        assertNull(AttributeSupport.getXMLBase(createdElement),
                "xml:base found erroneously (test setup failure)");
        AttributeSupport.addXMLBase(createdElement, TEST_NS);
        assertEquals(AttributeSupport.getXMLBase(createdElement), TEST_NS, "addXMLBase failed");

    }

    @Test public void testXMLSpace() {
        assertEquals(AttributeSupport.getXMLSpace(goodBaseIdSpaceLang), XMLSpace.DEFAULT, "xml:space mismatch");
        assertNull(AttributeSupport.getXMLSpace(noBaseIdSpaceLang), "xml:space found erroneously");
        assertNull(AttributeSupport.getXMLSpace(badSpace), "xml:space found erroneously");

        assertEquals(AttributeSupport.getXMLSpace(preserveSpace), XMLSpace.PRESERVE, "xml:space mismatch");

        // test Add
        boolean thrown = false;
        try {
            AttributeSupport.addXMLSpace(createdElement, nullValue());
        } catch (ConstraintViolationException e) {
            thrown = true;
        }
        assertTrue(thrown, "null string to addXMLBase");

        thrown = false;
        try {
            AttributeSupport.addXMLSpace(nullValue(), XMLSpace.DEFAULT);
        } catch (ConstraintViolationException e) {
            thrown = true;
        }
        assertTrue(thrown, "null element to addXMLSpace");

        assertNull(AttributeSupport.getXMLSpace(createdElement),
                "xml:space found erroneously (test setup failure)");
        AttributeSupport.addXMLSpace(createdElement, XMLSpace.DEFAULT);
        assertEquals(AttributeSupport.getXMLSpace(createdElement), XMLSpace.DEFAULT, "addXMLSpace failed");
    }

    @Test public void testXMLLang() {
        assertEquals(AttributeSupport.getXMLLang(goodBaseIdSpaceLang), "fr-ca", "xml:lang mismatch");
        assertNull(AttributeSupport.getXMLLang(noBaseIdSpaceLang), "xml:lang found erroneously");

        // test Add
        boolean thrown = false;
        try {
            AttributeSupport.addXMLLang(createdElement, nullValue());
        } catch (ConstraintViolationException e) {
            thrown = true;
        }
        assertTrue(thrown, "null string to addXMLBase");

        thrown = false;
        try {
            AttributeSupport.addXMLLang(nullValue(), "fr");
        } catch (ConstraintViolationException e) {
            thrown = true;
        }
        assertTrue(thrown, "null element to addXMLLang");

        assertNull(AttributeSupport.getXMLLang(createdElement),
                "xml:space found erroneously (test setup failure)");
        AttributeSupport.addXMLLang(createdElement, "fr");
        assertEquals(AttributeSupport.getXMLLang(createdElement), "fr", "addXMLLang failed");
    }

    @Test public void testGetID() {
        assertNull(AttributeSupport.getIdAttribute(createdElement), "ID of non id'd element is null");

        final Attr attr = AttributeSupport.getIdAttribute((Element) createdElement.getFirstChild());
        assert attr != null;
        assertEquals(attr.getValue(), TEST_ID_ATTRIBUTE_VALUE, "ID Attribute value mismatch");
        assertEquals(attr.getName(), TEST_ID_PREFIXEDATTRIBUTE, "ID Attribute name mismatch");
        assertEquals(attr.getNamespaceURI(), TEST_NS, "ID Attribute namespace mismatch");

    }

    @Test public void testHasAttribute() {
        assertFalse(AttributeSupport.hasAttribute(createdElement, idAttrQName), "Attribute lookup by QName");

        assertTrue(AttributeSupport.hasAttribute(goodBaseIdSpaceLang, XMLConstants.XML_BASE_ATTRIB_NAME),
                "attribute lookup by QName from file");
        assertFalse(AttributeSupport.hasAttribute(noBaseIdSpaceLang, XMLConstants.XML_BASE_ATTRIB_NAME),
                "attribute lookup by QName from file");

        Element child = (Element) createdElement.getFirstChild();
        assertTrue(AttributeSupport.hasAttribute(child, idAttrQName),
                "attribute lookup by QName in created element");
        assertTrue(
                AttributeSupport.hasAttribute(child, new QName(TEST_NS, TEST_ID_ATTRIBUTE, "xx" + TEST_PREFIX)),
                "attribute lookup by QName with changed prefix in created element");
        assertFalse(
                AttributeSupport.hasAttribute(child, new QName(TEST_NS + "/f", TEST_ID_ATTRIBUTE, TEST_PREFIX)),
                "attribute lookup by QName with changed NS in created element");
    }

    @Test(dependsOnMethods = {"testHasAttribute"}) public void testConstructAttribute() {
        assertFalse(AttributeSupport.hasAttribute(createdElement, idAttrQName), "precondition");
        createdElement.setAttributeNode(AttributeSupport.constructAttribute(document, idAttrQName));
        assertTrue(AttributeSupport.hasAttribute(createdElement, idAttrQName), "test constructAttribute(QName)");

        QName testQName = new QName(TEST_NS, TEST_ID_ATTRIBUTE + "XX", TEST_PREFIX);
        assertFalse(AttributeSupport.hasAttribute(createdElement, testQName), "precondition");
        createdElement.setAttributeNode(AttributeSupport.constructAttribute(document, TEST_NS,
                TEST_ID_ATTRIBUTE + "XX", TEST_PREFIX));
        assertTrue(AttributeSupport.hasAttribute(createdElement, testQName), "test constructAttribute(QName)");
    }

    @Test public void testRemoveAttribute() {
        assertFalse(AttributeSupport.removeAttribute(createdElement, idAttrQName), "Attribute remove by QName");
        Element child = (Element) createdElement.getFirstChild();
        assertTrue(AttributeSupport.removeAttribute(child, idAttrQName),
                "remove lookup by QName in created element");
        assertFalse(AttributeSupport.hasAttribute(child, idAttrQName),
                "attribute lookup by QName after it has been removed");
    }

    @Test public void testGetAttributeMethods() {
        // getAttribute(Element, QName)
        assertNull(AttributeSupport.getAttribute(noBaseIdSpaceLang, XMLConstants.XML_ID_ATTRIB_NAME),
                "no xml:id (lookup by QName)");
        final Attr attr = AttributeSupport.getAttribute(goodBaseIdSpaceLang, XMLConstants.XML_ID_ATTRIB_NAME);
        assert attr != null;
        assertEquals(attr.getValue(), "identifierGoodBaseIdSpaceLang",
                "Should have found correct attribute by value for xml_id attribute");

        // getAttributeValue(Element, QName)
        assertNull(AttributeSupport.getAttributeValue(noBaseIdSpaceLang, XMLConstants.XML_ID_ATTRIB_NAME),
                "no xml:id (lookup value by QName)");
        assertEquals(AttributeSupport.getAttributeValue(goodBaseIdSpaceLang, XMLConstants.XML_ID_ATTRIB_NAME),
                "identifierGoodBaseIdSpaceLang", "Should have found correct value for xml:id attribute by QName");

        // getAttributeValue(Element, String, String)
        assertNull(AttributeSupport.getAttributeValue(badSpace, XMLConstants.XML_NS, ""),
                "no value lookup with empty name)");
        assertNull(AttributeSupport.getAttributeValue(noBaseIdSpaceLang, XMLConstants.XML_NS, "space"),
                "no xml:space (lookup value by name)");
        assertEquals(AttributeSupport.getAttributeValue(badSpace, XMLConstants.XML_NS, "space"), "wibble",
                "Should have found correct value for xml:space attribute by name");

        // getAttributeValueAsBoolean(Attribute)
        // Use the previously tested AttributeSupport.getAttribute
        
        Attr reqAttr = AttributeSupport.getAttribute(attributes, new QName(TEST_NS, "testAttrEmpty"));
        assert reqAttr != null;
        assertNull(AttributeSupport.getAttributeValueAsBoolean(reqAttr), "\"\" should be null");

        reqAttr = AttributeSupport.getAttribute(attributes, new QName(TEST_NS, "testAttrZero"));
        assert reqAttr != null;
        Boolean flag = AttributeSupport.getAttributeValueAsBoolean(reqAttr); 
        assert flag != null;
        assertFalse(flag, "0 should be false");

        reqAttr = AttributeSupport.getAttribute(attributes, new QName(TEST_NS, "testAttrOne"));
        assert reqAttr != null;
        flag = AttributeSupport.getAttributeValueAsBoolean(reqAttr);
        assert flag != null;
        assertTrue(flag, "1 should be true");
        
        reqAttr = AttributeSupport.getAttribute(attributes, new QName(TEST_NS, "testAttrTwo"));
        assert reqAttr != null;
        assertNull(AttributeSupport.getAttributeValueAsBoolean(reqAttr), "2 should be null");
        
        reqAttr = AttributeSupport.getAttribute(attributes, new QName(TEST_NS, "testAttrFalse"));
        assert reqAttr != null;
        flag = AttributeSupport.getAttributeValueAsBoolean(reqAttr);
        assert flag != null;
        assertFalse(flag, "false should be false");
        
        reqAttr = AttributeSupport.getAttribute(attributes, new QName(TEST_NS, "testAttrTrue"));
        assert reqAttr != null;
        flag = AttributeSupport.getAttributeValueAsBoolean(reqAttr);
        assert flag != null;
        assertTrue(flag, "true should be true");
        
        reqAttr = AttributeSupport.getAttribute(attributes, new QName(TEST_NS, "testAttrTrueCaps"));
        assert reqAttr != null;
        assertNull(AttributeSupport.getAttributeValueAsBoolean(reqAttr), "TRUE should be null");

        // getAttributeValueAsList(Attribute)
        // Use the previously tested AttributeSupport.getAttribute
        
        reqAttr = AttributeSupport.getAttribute(attributes, new QName(TEST_NS, "testAttrEmpty"));
        assert reqAttr != null;
        assertTrue(AttributeSupport.getAttributeValueAsList(reqAttr).isEmpty(),
                "\"\" attribute should give empty list");
        
        reqAttr = AttributeSupport.getAttribute(attributes, new QName(TEST_NS, "testAttrZero"));
        assert reqAttr != null;
        assertEquals(AttributeSupport.getAttributeValueAsList(reqAttr), Arrays.asList("0"), "attribute called testAttrZero");
        
        reqAttr = AttributeSupport.getAttribute(attributes, new QName(TEST_NS, "testAttrList"));
        assert reqAttr != null;
        assertEquals(AttributeSupport.getAttributeValueAsList(reqAttr), Arrays.asList("0", "1", "2", "3", "4", "5", "6"),
                "attribute called testAttrList");

        // getAttributeValueAsQName(Attribute)
        // Use the previously tested AttributeSupport.getAttribute
        
        reqAttr = AttributeSupport.getAttribute(attributes, new QName(TEST_NS, "testAttrEmpty"));
        assert reqAttr != null;
        assertNull(AttributeSupport.getAttributeValueAsQName(reqAttr), "\"\" should be null");
        
        reqAttr = AttributeSupport.getAttribute(attributes, new QName(TEST_NS, "testAttrQName"));
        assert reqAttr != null;
        assertEquals(AttributeSupport.getAttributeValueAsQName(reqAttr), idAttrQName, "attribute called testAttrQName");
        
        reqAttr = AttributeSupport.getAttribute(attributes, new QName(TEST_NS, "testAttrZero"));
        assert reqAttr != null;
        assertEquals(AttributeSupport.getAttributeValueAsQName(reqAttr), new QName("0"), "attribute called testAttrZero");

        // getDateTimeAttribute
        // Use the previously tested AttributeSupport.getAttribute
        
        reqAttr = AttributeSupport.getAttribute(attributes, new QName(TEST_NS, "testAttrEmpty"));
        assert reqAttr != null;
        assertNull(AttributeSupport.getDateTimeAttribute(reqAttr), "\"\" should be null");
        
        reqAttr = AttributeSupport.getAttribute(attributes, new QName(TEST_NS, "testAttrEmpty"));
        assert reqAttr != null;
        assertNull(AttributeSupport.getDateTimeAttribute(reqAttr), "\"0\" should be null");
        
        reqAttr = AttributeSupport.getAttribute(attributes, new QName(TEST_NS, "testAttrEpochPlusOneSec"));
        assert reqAttr != null;
        assertEquals(
                AttributeSupport.getDateTimeAttribute(reqAttr),
                Instant.ofEpochSecond(1), "attribute called testAttrEpochPlusOneSec");

        // getDurationAttributeValueAsLong
        // Use the previously tested AttributeSupport.getAttribute

        reqAttr = AttributeSupport.getAttribute(attributes, new QName(TEST_NS, "testAttrEmpty"));
        assert reqAttr != null;
        assertNull(AttributeSupport.getDurationAttributeValue(reqAttr), "\"\" should be null");
        
        reqAttr = AttributeSupport.getAttribute(attributes, new QName(TEST_NS, "testAttrEmpty"));
        assert reqAttr != null;
        assertNull(AttributeSupport.getDurationAttributeValue(reqAttr), "\"0\" should be null");
        
        reqAttr = AttributeSupport.getAttribute(attributes, new QName(TEST_NS, "testAttrMinusOneDay"));
        assert reqAttr != null;
        assertEquals(
                AttributeSupport.getDurationAttributeValue(reqAttr),
                Duration.ofDays(-1), "attribute called testAttrMinusOneDay");
    }

    @Test(dependsOnMethods = {"testGetAttributeMethods", "testGetID"}) public void testAppends() {
        String qNameBase = "name";
        String testResult = TEST_ID_ATTRIBUTE_VALUE;

        QName qName = new QName(TEST_NS, qNameBase, TEST_PREFIX);

        assertNull(AttributeSupport.getAttributeValue(createdElement, qName), "Test precondition");
        boolean thrown = false;
        try {
            AttributeSupport.appendAttribute(nullValue(), qName, testResult);
        } catch (ConstraintViolationException e) {
            thrown = true;
        }
        assertTrue(thrown, "null element should throw");

        thrown = false;
        try {
            AttributeSupport.appendAttribute(createdElement, nullValue(), testResult);
        } catch (ConstraintViolationException e) {
            thrown = true;
        }
        assertTrue(thrown, "null qname should throw");

        thrown = false;
        try {
            AttributeSupport.appendAttribute(createdElement, qName, nullValue());
        } catch (ConstraintViolationException e) {
            thrown = true;
        }
        assertTrue(thrown, "null string should throw");

        thrown = false;
        try {
            AttributeSupport.appendAttribute(createdElement, qName, testResult);
        } catch (ConstraintViolationException e) {
            thrown = true;
        }
        assertFalse(thrown, "All non nulls should not throw");
        assertEquals(AttributeSupport.getAttributeValue(createdElement, qName), testResult,
                "appendAttribute(Element, QName, String) failed");

        // appendAttribute(Element, QName, String, boolean)
        qNameBase = qNameBase + "New";
        testResult = testResult + "New";
        qName = new QName(TEST_NS, qNameBase, TEST_PREFIX);
        assertNull(AttributeSupport.getAttributeValue(createdElement, qName), "Test precondition");
        assertNull(AttributeSupport.getIdAttribute(createdElement), "Test precondition");

        thrown = false;
        try {
            AttributeSupport.appendAttribute(nullValue(), qName, testResult, false);
        } catch (ConstraintViolationException e) {
            thrown = true;
        }
        assertTrue(thrown, "null element should throw");

        thrown = false;
        try {
            AttributeSupport.appendAttribute(createdElement, nullValue(), testResult, false);
        } catch (ConstraintViolationException e) {
            thrown = true;
        }
        assertTrue(thrown, "null qname should throw");

        thrown = false;
        try {
            AttributeSupport.appendAttribute(createdElement, qName, (String) nullValue(), false);
        } catch (ConstraintViolationException e) {
            thrown = true;
        }
        assertTrue(thrown, "null string should throw");

        thrown = false;
        try {
            AttributeSupport.appendAttribute(createdElement, qName, testResult, false);
        } catch (ConstraintViolationException e) {
            thrown = true;
        }
        assertFalse(thrown, "All non nulls should not throw");
        assertEquals(AttributeSupport.getAttributeValue(createdElement, qName), testResult,
                "appendAttribute(Element, QName, String) failed");
        assertNull(AttributeSupport.getIdAttribute(createdElement), "Should not have added an id Attribute");

        qNameBase = qNameBase + "New";
        testResult = testResult + "New";
        qName = new QName(TEST_NS, qNameBase, TEST_PREFIX);
        assertNull(AttributeSupport.getAttributeValue(createdElement, qName), "Test precondition");
        assertNull(AttributeSupport.getIdAttribute(createdElement), "Test precondition");
        AttributeSupport.appendAttribute(createdElement, qName, testResult, true);
        
        Attr id = AttributeSupport.getIdAttribute(createdElement);
        assert id != null;
        assertEquals(id.getValue(), testResult,
                "id Attribute added correctly");
        AttributeSupport.removeAttribute(createdElement, qName);

        // appendAttribute(Element, QName, List<String>, boolean)
        qNameBase = qNameBase + "New";
        List<String> data = Arrays.asList("one", "2", "iii");
        testResult = "one 2 iii";
        qName = new QName(TEST_NS, qNameBase, TEST_PREFIX);
        assertNull(AttributeSupport.getAttributeValue(createdElement, qName), "Test precondition");
        assertNull(AttributeSupport.getIdAttribute(createdElement), "Test precondition");

        thrown = false;
        try {
            AttributeSupport.appendAttribute(nullValue(), qName, data, false);
        } catch (ConstraintViolationException e) {
            thrown = true;
        }
        assertTrue(thrown, "null element should throw");

        thrown = false;
        try {
            AttributeSupport.appendAttribute(createdElement, nullValue(), data, false);
        } catch (ConstraintViolationException e) {
            thrown = true;
        }
        assertTrue(thrown, "null qname should throw");

        thrown = false;
        try {
            AttributeSupport.appendAttribute(createdElement, qName, data, false);
        } catch (ConstraintViolationException e) {
            thrown = true;
        }
        assertFalse(thrown, "All non nulls should not throw");
        assertEquals(AttributeSupport.getAttributeValue(createdElement, qName), testResult,
                "appendAttribute(Element, QName, String) failed");
        assertNull(AttributeSupport.getIdAttribute(createdElement), "Should not have added an id Attribute");

        qNameBase = qNameBase + "New";
        qName = new QName(TEST_NS, qNameBase, TEST_PREFIX);
        assertNull(AttributeSupport.getAttributeValue(createdElement, qName), "Test precondition");
        assertNull(AttributeSupport.getIdAttribute(createdElement), "Test precondition");
        AttributeSupport.appendAttribute(createdElement, qName, data, true);
        
        id = AttributeSupport.getIdAttribute(createdElement);
        assert id != null;
        assertEquals(id.getValue(), testResult, "id Attribute added correctly");

        final var duration = Duration.ofSeconds(1);
        qNameBase = qNameBase + "New";
        qName = new QName(TEST_NS, qNameBase, TEST_PREFIX);
        assertNull(AttributeSupport.getAttributeValue(createdElement, qName), "Test precondition");
        thrown = false;
        try {
            AttributeSupport.appendDurationAttribute(nullValue(), qName, duration);
        } catch (ConstraintViolationException e) {
            thrown = true;
        }
        assertTrue(thrown, "null element should throw");

        thrown = false;
        try {
            AttributeSupport.appendDurationAttribute(createdElement, nullValue(), duration);
        } catch (ConstraintViolationException e) {
            thrown = true;
        }
        assertTrue(thrown, "null qname should throw");

        thrown = false;
        try {
            AttributeSupport.appendDurationAttribute(createdElement, qName, duration);
        } catch (ConstraintViolationException e) {
            thrown = true;
        }
        assertFalse(thrown, "All non nulls should not throw");
        assertEquals(
                AttributeSupport.getDurationAttributeValue(
                        Constraint.isNotNull(AttributeSupport.getAttribute(createdElement, qName), "NPE")),
                        duration, "getDurationAttributeValueAsLong failed");

        // Construct a time that contains nothing below the level of milliseconds,
        // for compatibility with representations that don't have higher precision.
        final var time = Instant.now().truncatedTo(ChronoUnit.MILLIS);
        qNameBase = qNameBase + "New";
        qName = new QName(TEST_NS, qNameBase, TEST_PREFIX);
        assertNull(AttributeSupport.getAttributeValue(createdElement, qName), "Test precondition");
        thrown = false;
        try {
            AttributeSupport.appendDateTimeAttribute(nullValue(), qName, time);
        } catch (ConstraintViolationException e) {
            thrown = true;
        }
        assertTrue(thrown, "null element should throw");

        thrown = false;
        try {
            AttributeSupport.appendDateTimeAttribute(createdElement, nullValue(), time);
        } catch (ConstraintViolationException e) {
            thrown = true;
        }
        assertTrue(thrown, "null qname should throw");

        thrown = false;
        try {
            AttributeSupport.appendDateTimeAttribute(createdElement, qName, time);
        } catch (ConstraintViolationException e) {
            thrown = true;
        }
        assertFalse(thrown, "All non nulls should not throw");
        assertEquals(
                AttributeSupport.getDateTimeAttribute(
                        Constraint.isNotNull(AttributeSupport.getAttribute(createdElement, qName), "NPE")),
                        time, "getDurationAttributeValueAsLong failed");

    }

    @Test
    public void testDomAssumptions() {
        /*
         * Test some assumptions we want to make in Spring Parsing
         */
        assert attributes.hasAttributeNS(TEST_NS, "testAttrEmpty");
        assert !attributes.hasAttributeNS(TEST_NS, "testAttrNonExist");

        assert attributes.getAttributeNodeNS(TEST_NS, "testAttrEmpty") != null;
        assert attributes.getAttributeNodeNS(TEST_NS, "testAttrNonExist") == null;

        assert "".equals(attributes.getAttributeNS(TEST_NS, "testAttrEmpty"));
        //
        // This case is weird
        //
        String s= attributes.getAttributeNS(TEST_NS, "testAttrNonExist");
        assert "".equals(s);
        s = AttributeSupport.getAttributeValue(attributes, TEST_NS, "testAttrNonExist");
        assert s == null;
    }
    
    @Test
    public void testQNameContent() throws XMLParserException {
        final DocumentBuilder builder = parserPool.getBuilder();
        final Document doc = builder.newDocument();
        final Attr attr = doc.createAttributeNS("https://example", "foo");
        attr.setNodeValue("foo");
        
        QName qname = AttributeSupport.getAttributeValueAsQName(attr);
        assert qname != null;
        assertEquals(qname.getLocalPart(), "foo");
        assertEquals(qname.getNamespaceURI(), "");
        assertEquals(qname.getPrefix(), "");

        attr.setNodeValue("bar:foo");
        qname = AttributeSupport.getAttributeValueAsQName(attr);
        assert qname != null;
        assertEquals(qname.getLocalPart(), "foo");
        assertEquals(qname.getNamespaceURI(), "");
        assertEquals(qname.getPrefix(), "bar");

        attr.setNodeValue("bar:foo:baz");
        try {
            qname = AttributeSupport.getAttributeValueAsQName(attr);
            fail("Expected IllegalStateException");
        } catch (final IllegalStateException e) {
            // expected
        }
    }

    private <T> T nullValue() {
        return null;
    }
    
}