/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.xml;

import java.io.IOException;
import java.io.InputStream;
import java.time.Duration;
import java.time.Instant;

import javax.xml.namespace.QName;
import javax.xml.parsers.DocumentBuilder;

import org.testng.Assert;
import org.testng.annotations.BeforeClass;
import org.testng.annotations.Test;
import org.w3c.dom.Element;
import org.xml.sax.SAXException;

import net.shibboleth.shared.annotation.constraint.NonnullBeforeTest;
import net.shibboleth.shared.component.ComponentInitializationException;
import net.shibboleth.shared.xml.impl.BasicParserPool;

/**
 * Tests for {@link DOMTypeSupport};
 */
@SuppressWarnings("javadoc")
public class DOMTypeSupportTest {

    @NonnullBeforeTest private ParserPool parserPool;
    @NonnullBeforeTest private Element xsStringXSITypeElement;
    @NonnullBeforeTest private Element noXSITypeElement;
    
    @BeforeClass public void setup() throws ComponentInitializationException, SAXException, IOException, XMLParserException {
        final BasicParserPool pool = new BasicParserPool();
        pool.initialize();
        parserPool = pool;
        
        final DocumentBuilder builder = parserPool.getBuilder();
        assert builder!= null;
        try (final InputStream s = getClass().getResourceAsStream("/net/shibboleth/shared/xml/getXSIType.xml")) {
            xsStringXSITypeElement = (Element) builder.parse(s).getFirstChild();
            assert xsStringXSITypeElement!=null;
        }

        try (final InputStream s = getClass().getResourceAsStream("/net/shibboleth/shared/xml/noXSIType.xml")) {
            noXSITypeElement = (Element) builder.parse(s).getFirstChild();
            assert noXSITypeElement!=null;
        }
        
        parserPool.returnBuilder(builder);
    }
    
    
    @Test public void testInstantToString() {
        Assert.assertEquals(DOMTypeSupport.instantToString(Instant.EPOCH.plusMillis(1000)), "1970-01-01T00:00:01.000Z", "Epoch plus one second");
        Assert.assertEquals(DOMTypeSupport.instantToString(Instant.EPOCH.plusMillis(-1000)), "1969-12-31T23:59:59.000Z", "Epoch minus one second");
    }

    @Test public void testStringToDuration() {
        Assert.assertEquals(DOMTypeSupport.stringToDuration("P0Y0M0DT00H00M01S"), Duration.ofSeconds(1), "One second duration");
        Assert.assertEquals(DOMTypeSupport.stringToDuration("-P1D"), Duration.ofDays(-1), "Back One day duration");
    }

    @Test public void testDurationToString() {
        // We have to check for two different possible return values because Oracle's and Xerces' implementations
        // are different.
        
        String onesec = DOMTypeSupport.durationToString(Duration.ofSeconds(1));
        Assert.assertTrue("P0Y0M0DT0H0M1.000S".equals(onesec) || "PT1.000S".equals(onesec), "One second duration");

        String backday = DOMTypeSupport.durationToString(Duration.ofDays(-1));
        Assert.assertTrue("-P0Y0M1DT0H0M0.000S".equals(backday) || "-P1DT0H0M0.000S".equals(backday), "Back one day duration");
    }
    
    @Test public void testGetXSIType() {
        Assert.assertEquals(DOMTypeSupport.getXSIType(xsStringXSITypeElement),
                new QName("http://www.w3.org/2001/XMLSchema", "string", "xs"),
                "XSI type clash");
        Assert.assertNull(DOMTypeSupport.getXSIType(noXSITypeElement), "No xsiType expected");
    }

    @Test public void testHasXSIType() {
        Assert.assertTrue(DOMTypeSupport.hasXSIType(xsStringXSITypeElement)," Expected xsi:type");
        Assert.assertFalse(DOMTypeSupport.hasXSIType(noXSITypeElement), "No xsiType expected");
    }

}
