/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.xml;

import java.io.IOException;
import java.io.InputStream;
import java.util.Arrays;
import java.util.List;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.xml.namespace.QName;
import javax.xml.parsers.DocumentBuilder;

import net.shibboleth.shared.annotation.constraint.NonnullBeforeTest;
import net.shibboleth.shared.annotation.constraint.NotEmpty;
import net.shibboleth.shared.component.ComponentInitializationException;
import net.shibboleth.shared.logic.ConstraintViolationException;
import net.shibboleth.shared.primitive.StringSupport;
import net.shibboleth.shared.xml.impl.BasicParserPool;

import org.testng.Assert;
import org.testng.annotations.BeforeClass;
import org.testng.annotations.Test;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.SAXException;

/**
 * Test {@link ElementSupport}. These tests use {@link Test#dependsOnMethods()} to ensure that we suppress a test if
 * functionality it relies on has not been tested correctly. This avoids false failures.
 */
@SuppressWarnings("javadoc")
public class ElementSupportTest {

    @Nonnull @NotEmpty private static final String TEST_NS = "http://example.org/NameSpace";

    @Nonnull @NotEmpty private static final String OTHER_NS = "http://example.org/OtherSpace";

    @Nonnull @NotEmpty private static final String TEST_PREFIX = "testns";

    @Nonnull @NotEmpty private static final String TEST_ELEMENT_NAME = "Element1";

    @Nonnull @NotEmpty private static final String ROOT_ELEMENT = "Container";

    @Nonnull private static final QName TEST_ELEMENT_QNAME = new QName(TEST_NS, TEST_ELEMENT_NAME, TEST_PREFIX);

    @NonnullBeforeTest private BasicParserPool parserPool;

    @NonnullBeforeTest private Document testFileDocument;

    @NonnullBeforeTest private Document testerDocument;

    @NonnullBeforeTest private Element rootElement;

    @BeforeClass public void setUp() throws XMLParserException, ComponentInitializationException, SAXException,
            IOException {
        parserPool = new BasicParserPool();
        parserPool.setCoalescing(false);
        
        parserPool.initialize();

        DocumentBuilder builder = parserPool.getBuilder();
        try (final InputStream s = getClass().getResourceAsStream("/net/shibboleth/shared/xml/elementSupportTest.xml")) {
            testFileDocument = builder.parse(s);
            rootElement = (Element) testFileDocument.getFirstChild();
            testerDocument = builder.newDocument();
        } finally {
            parserPool.returnBuilder(builder);
        }
        assert rootElement!=null && testerDocument!=null && testFileDocument!=null;
    }

    @Test public void testIsElementNamed() {
        Assert.assertFalse(ElementSupport.isElementNamed(rootElement, null, ROOT_ELEMENT),
                "not find if provided namespace is null");
        Assert.assertTrue(ElementSupport.isElementNamed(rootElement, new QName(TEST_NS, ROOT_ELEMENT, TEST_PREFIX)),
                "lookup against QNAME");
        Assert.assertTrue(ElementSupport.isElementNamed(rootElement, TEST_NS, ROOT_ELEMENT), "lookup against name");

        Assert.assertFalse(ElementSupport.isElementNamed(rootElement, TEST_NS, ROOT_ELEMENT.toUpperCase()),
                "lookup against upper of name");
        Assert.assertFalse(ElementSupport.isElementNamed(rootElement, TEST_NS.toUpperCase(), ROOT_ELEMENT),
                "lookup against upper of namespace");

    }

    @Test(dependsOnMethods = {"testIsElementNamed"}) public void testGetChildElements() {
        Assert.assertEquals(ElementSupport.getChildElements(rootElement).size(), 8, "unnanmed element lookup");
        List<Element> list = ElementSupport.getChildElements(rootElement, TEST_ELEMENT_QNAME);
        Assert.assertEquals(list.size(), 3, "Named element lookup");
        for (Element e : list) {
            Assert.assertTrue(ElementSupport.isElementNamed(e, TEST_ELEMENT_QNAME));
        }
    }

    @Test(dependsOnMethods = {"testIsElementNamed"}) public void testGetChildElementsByTagName() {
        Assert.assertTrue(ElementSupport.getChildElementsByTagNameNS(rootElement, null, TEST_ELEMENT_NAME).isEmpty(),
                "getChildElementsByTagNameNS: Null name space should provide empty list");

        List<Element> list = ElementSupport.getChildElementsByTagName(rootElement, TEST_ELEMENT_NAME);
        Assert.assertEquals(list.size(), 5, "getChildElementsByTagName size");
        int i = 0;
        for (Element e : list) {
            if (ElementSupport.isElementNamed(e, TEST_ELEMENT_QNAME)) {
                i++;
            }
        }
        Assert.assertEquals(i, 3, "getChildElementsByTagName size");

        list = ElementSupport.getChildElementsByTagNameNS(rootElement, TEST_NS, TEST_ELEMENT_NAME);
        Assert.assertEquals(list.size(), 3, "getChildElementsByTagName size");
        for (Element e : list) {
            Assert.assertTrue(ElementSupport.isElementNamed(e, TEST_ELEMENT_QNAME));
        }

    }

    @Test(dependsOnMethods = {"testGetChildElementsByTagName"}) public void testGetElementAncestor() {
        Assert.assertNull(ElementSupport.getElementAncestor(rootElement),
                "getElementAncestor: root node should provide null result");

        Element child = ElementSupport.getChildElementsByTagName(rootElement, "Element4").get(0);
        Element grandChild = ElementSupport.getChildElementsByTagName(child, "Element1").get(0);

        Assert.assertEquals(ElementSupport.getElementAncestor(child), rootElement, "getElementAncestor for child");
        Assert.assertEquals(ElementSupport.getElementAncestor(grandChild), child, "getElementAncestor for grand child");

    }

    @Test(dependsOnMethods = {"testGetChildElementsByTagName"}) public void testGetElementContentAsString() {
        final String empty = StringSupport.trim(ElementSupport.getElementContentAsString(rootElement));
        assert empty != null;
        Assert.assertTrue(empty.isEmpty(),
                "getElementContentAsList: Empty element should provide empty result");

        Element interesting =
                ElementSupport.getChildElementsByTagName(
                        ElementSupport.getChildElementsByTagName(rootElement, "Element4").get(0), "Element1").get(0);

        Assert.assertEquals(ElementSupport.getElementContentAsString(interesting), "Some Random foo" + "\n" + "test");
    }

    @Test(dependsOnMethods = {"testGetChildElementsByTagName"}) public void testGetElementContentAsList() {
        Assert.assertTrue(ElementSupport.getElementContentAsList(rootElement).isEmpty(),
                "getElementContentAsList: Empty element should provide empty result");

        Element interesting =
                ElementSupport.getChildElementsByTagName(
                        ElementSupport.getChildElementsByTagName(rootElement, "Element4").get(0), "Element1").get(0);

        Assert.assertEquals(ElementSupport.getElementContentAsList(interesting),
                Arrays.asList("Some", "Random", "foo", "test"));

    }

    @Test(dependsOnMethods = {"testGetChildElementsByTagName"}) public void testGetElementContentAsQName() {
        Assert.assertNull(ElementSupport.getElementContentAsQName(rootElement),
                "getElementContentAsQName: Empty element should provide empty result");

        Element parent = ElementSupport.getChildElementsByTagName(rootElement, "Element4").get(0);
        List<Element> children = ElementSupport.getChildElementsByTagName(parent, "QName");

        Assert.assertEquals(ElementSupport.getElementContentAsQName(children.get(0)), new QName(OTHER_NS, "localname"));
        Assert.assertNull(ElementSupport.getElementContentAsQName(children.get(1)),
                "getElementContentAsQName: invalid qname should return null");
    }

    @Test(dependsOnMethods = {"testIsElementNamed"}) public void testGetChildAndNext() {
        Element element = ElementSupport.getFirstChildElement(rootElement);
        assert element != null;
        Assert.assertTrue(ElementSupport.isElementNamed(element, TEST_ELEMENT_QNAME), "getFirstChildElement");
        Assert.assertNull(ElementSupport.getFirstChildElement(element),
                "getFirstChildElement: Empty element should provide null result");

        element = ElementSupport.getNextSiblingElement(element);
        assert element != null;
        Assert.assertTrue(ElementSupport.isElementNamed(element, TEST_ELEMENT_QNAME), "getNextSiblingElement 1");

        element = ElementSupport.getNextSiblingElement(element);
        QName qName = new QName(TEST_NS, "Element2", TEST_PREFIX);
        assert element != null;
        Assert.assertTrue(ElementSupport.isElementNamed(element, qName), "getNextSiblingElement 2");

        element = ElementSupport.getNextSiblingElement(element);
        qName = new QName(TEST_NS, TEST_ELEMENT_NAME, "mynsagain");
        assert element != null;
        Assert.assertTrue(ElementSupport.isElementNamed(element, qName), "getNextSiblingElement 3");

        element = ElementSupport.getNextSiblingElement(element);
        qName = new QName(OTHER_NS, TEST_ELEMENT_NAME, TEST_PREFIX);
        assert element != null;
        Assert.assertTrue(ElementSupport.isElementNamed(element, qName), "getNextSiblingElement 4 ");

        element = ElementSupport.getNextSiblingElement(element);
        assert element != null;
        Assert.assertTrue(ElementSupport.isElementNamed(element, qName), "getNextSiblingElement 5");

        element = ElementSupport.getNextSiblingElement(element);
        qName = new QName(OTHER_NS, "Element2", TEST_PREFIX);
        assert element != null;
        Assert.assertTrue(ElementSupport.isElementNamed(element, qName), "getNextSiblingElement 6");

        element = ElementSupport.getNextSiblingElement(element);
        qName = new QName(TEST_NS, "Element4", TEST_PREFIX);
        assert element != null;
        Assert.assertTrue(ElementSupport.isElementNamed(element, qName), "getNextSiblingElement 7");

        Assert.assertNull(ElementSupport.getNextSiblingElement(element),
                "getNextSiblingElement: final element should provide null result");
    }

    @Test public void testGetIndexedChildElements() {
        Map<QName, List<Element>> map = ElementSupport.getIndexedChildElements(rootElement);

        Assert.assertEquals(map.get(TEST_ELEMENT_QNAME).size(), 3, "getIndexedChildElements for " + TEST_ELEMENT_QNAME);

        QName qname = new QName(TEST_NS, "Element2", TEST_PREFIX);
        Assert.assertEquals(map.get(qname).size(), 1, "getIndexedChildElements for " + qname);

        qname = new QName(OTHER_NS, "Element1", "otherns");
        Assert.assertEquals(map.get(qname).size(), 2, "getIndexedChildElements for " + qname);

        qname = new QName(OTHER_NS, "Element2", "otherns");
        Assert.assertEquals(map.get(qname).size(), 1, "getIndexedChildElements for " + qname);
        Element elementNoKids = map.get(qname).get(0);

        qname = new QName(TEST_NS, "Element4", TEST_PREFIX);
        Assert.assertEquals(map.get(qname).size(), 1, "getIndexedChildElements for " + qname);

        Assert.assertTrue(ElementSupport.getIndexedChildElements(elementNoKids).isEmpty(),
                "getIndexedChildElements with a no child element");

    }

    @Test(dependsOnMethods = {"testConstructElement"}) public void testAdoptElement() throws XMLParserException {
        DocumentBuilder builder = parserPool.getBuilder();
        try {
            Document otherDocument = builder.newDocument();
            Element element = ElementSupport.constructElement(testerDocument, TEST_ELEMENT_QNAME);
            testerDocument.appendChild(element);

            //
            // Failure conditions
            //
            boolean thrown = false;
            try {
                ElementSupport.adoptElement(nullValue(), element);
            } catch (ConstraintViolationException e) {
                thrown = true;
            }
            Assert.assertTrue(thrown, "adoptElement: Null Document should assert");

            thrown = false;
            try {
                ElementSupport.adoptElement(testerDocument, nullValue());
            } catch (ConstraintViolationException e) {
                thrown = true;
            }
            Assert.assertTrue(thrown, "adoptElement: Null Element should assert");

            //
            // And success
            //
            Assert.assertNotSame(element.getOwnerDocument(), otherDocument, "Initially: element not owned");
            Assert.assertEquals(element.getOwnerDocument(), testerDocument, "Initially: element owned");
            ElementSupport.adoptElement(testerDocument, element);
            Assert.assertNotSame(element.getOwnerDocument(), otherDocument, "ReAdopt: element not owned");
            Assert.assertEquals(element.getOwnerDocument(), testerDocument, "ReAdopt: element owned");
            ElementSupport.adoptElement(otherDocument, element);
            Assert.assertEquals(element.getOwnerDocument(), otherDocument, "After: element owned");
            Assert.assertNotSame(element.getOwnerDocument(), testerDocument, "After:  new element not owned");

        } finally {
            parserPool.returnBuilder(builder);
        }
    }

    @Test(dependsOnMethods = {"testConstructElement", "testGetChildElements", "testGetChildElementsByTagName"}) public
            void testAppendChildElement() throws XMLParserException {
        DocumentBuilder builder = parserPool.getBuilder();
        try {
            Document myDocument = builder.newDocument();
            Element myRoot =
                    ElementSupport.constructElement(testerDocument, new QName(TEST_NS, ROOT_ELEMENT, TEST_PREFIX));
            Element testElement = ElementSupport.constructElement(testerDocument, TEST_ELEMENT_QNAME);
            boolean thrown = false;
            try {
                ElementSupport.appendChildElement(nullValue(), testElement);
            } catch (ConstraintViolationException e) {
                thrown = true;
            }
            Assert.assertTrue(thrown, "appendChildElement: assert on null parent");

            Assert.assertTrue(ElementSupport.getChildElements(myRoot).isEmpty(),
                    "appendChildElement: no elements to start with");
            ElementSupport.appendChildElement(myRoot, testElement);
            Assert.assertEquals(ElementSupport.getChildElements(myRoot).size(), 1,
                    "appendChildElement: One elements after one insertion");
            ElementSupport.appendChildElement(myRoot, testElement);
            Assert.assertEquals(ElementSupport.getChildElements(myRoot).size(), 1,
                    "appendChildElement: One elements after re insertion");

            QName qName = new QName(OTHER_NS, TEST_ELEMENT_NAME);
            ElementSupport.appendChildElement(myRoot, ElementSupport.constructElement(myDocument, TEST_ELEMENT_QNAME));
            ElementSupport.appendChildElement(myRoot, ElementSupport.constructElement(myDocument, qName));
            ElementSupport.appendChildElement(myRoot, ElementSupport.constructElement(myDocument, TEST_ELEMENT_QNAME));
            ElementSupport.appendChildElement(myRoot, ElementSupport.constructElement(myDocument, qName));
            Assert.assertEquals(ElementSupport.getChildElements(myRoot).size(), 5,
                    "appendChildElement:  elements after insertion");
            Assert.assertEquals(ElementSupport.getChildElementsByTagNameNS(myRoot, TEST_NS, TEST_ELEMENT_NAME).size(),
                    3, "appendChildElement:  elements after insertion");
            Assert.assertEquals(ElementSupport.getChildElementsByTagNameNS(myRoot, OTHER_NS, TEST_ELEMENT_NAME).size(),
                    2, "appendChildElement:  elements after insertion");
        } finally {
            parserPool.returnBuilder(builder);
        }
    }

    @Test(dependsOnMethods = {"testConstructElement"}) public void testAppendChildText() {
        Element element = ElementSupport.constructElement(testerDocument, TEST_ELEMENT_QNAME);
        Assert.assertNull(StringSupport.trimOrNull(element.getTextContent()),
                "appendTextContent: initially element has no text");
        boolean thrown = false;
        try {
            ElementSupport.appendTextContent(nullValue(), "test");
        } catch (ConstraintViolationException e) {
            thrown = true;
        }
        Assert.assertTrue(thrown, "appendTextContent: null element should assert");

        ElementSupport.appendTextContent(element, null);
        Assert.assertNull(StringSupport.trimOrNull(element.getTextContent()),
                "appendTextContent: After null assert element has no text");
        ElementSupport.appendTextContent(element, "");
        Assert.assertEquals(element.getTextContent().length(), 0, "appendTextContent: contents match even when empty");

        ElementSupport.appendTextContent(element, " test Text");
        Assert.assertEquals(element.getTextContent(), " test Text", "appendTextContent: contents match");

        ElementSupport.appendTextContent(element, " more Text");
        Assert.assertEquals(element.getTextContent(), " test Text more Text",
                "appendTextContent: contents after second append");
    }

    @Test(dependsOnMethods = {"testGetChildElementsByTagName"}) public void testConstructElement()
            throws XMLParserException {
        Element e = ElementSupport.constructElement(testerDocument, TEST_ELEMENT_QNAME);
        Assert.assertTrue(ElementSupport.isElementNamed(e, TEST_ELEMENT_QNAME));

        e = ElementSupport.constructElement(testerDocument, TEST_NS, TEST_ELEMENT_NAME, TEST_PREFIX);
        Assert.assertTrue(ElementSupport.isElementNamed(e, TEST_ELEMENT_QNAME));
    }

    @Test public void testConstructElementBadParms() throws XMLParserException {
        boolean thrown = false;
        try {
            ElementSupport.constructElement(nullValue(), TEST_ELEMENT_QNAME);
        } catch (ConstraintViolationException e) {
            thrown = true;
        }
        Assert.assertTrue(thrown, "constructElement(Document, QName): null Document should throw");

        thrown = false;
        try {
            ElementSupport.constructElement(testerDocument, nullValue());
        } catch (ConstraintViolationException e) {
            thrown = true;
        }
        Assert.assertTrue(thrown, "constructElement(Document, QName): null QName should throw");

        thrown = false;
        try {
            ElementSupport.constructElement(testerDocument, TEST_NS, nullValue(), TEST_PREFIX);
        } catch (ConstraintViolationException e) {
            thrown = true;
        }
        Assert.assertTrue(thrown, "constructElement(Document, String, String, String): null ElementName should throw");

        thrown = false;
        try {
            ElementSupport.constructElement(nullValue(), TEST_NS, TEST_ELEMENT_NAME, TEST_PREFIX);
        } catch (ConstraintViolationException e) {
            thrown = true;
        }
        Assert.assertTrue(thrown, "constructElement(Document, String, String, String): null ElementName should throw");

    }

    @Test(dependsOnMethods = {"testConstructElement", "testGetChildElements"}) public void testSetDocumentElement()
            throws XMLParserException {
        DocumentBuilder builder = parserPool.getBuilder();
        try {
            Document myDocument = builder.newDocument();
            Element myRoot =
                    ElementSupport.constructElement(testerDocument, new QName(TEST_NS, ROOT_ELEMENT, TEST_PREFIX));
            boolean thrown = false;
            try {
                ElementSupport.setDocumentElement(myDocument, nullValue());
            } catch (ConstraintViolationException e) {
                thrown = true;
            }
            Assert.assertTrue(thrown, "setDocumentElement: null Element should assert");

            thrown = false;
            try {
                ElementSupport.setDocumentElement(nullValue(), myRoot);
            } catch (ConstraintViolationException e) {
                thrown = true;
            }
            Assert.assertTrue(thrown, "setDocumentElement: null Document should assert");
            Assert.assertTrue(ElementSupport.getChildElements(myDocument).isEmpty(), "New document should be empty");

            ElementSupport.setDocumentElement(myDocument, myRoot);
            List<Element> list = ElementSupport.getChildElements(myDocument);
            Assert.assertEquals(list.size(), 1, "setDocumentElement: One child after element set");
            Assert.assertEquals(list.get(0), myRoot, "setDocumentElement: after element set");

            ElementSupport.setDocumentElement(myDocument, myRoot);
            list = ElementSupport.getChildElements(myDocument);
            Assert.assertEquals(list.size(), 1, "setDocumentElement: One child after element re-set");
            Assert.assertEquals(list.get(0), myRoot, "setDocumentElement: after element re-set");

            Element otherRoot =
                    ElementSupport.constructElement(testerDocument, new QName(TEST_NS, ROOT_ELEMENT, TEST_PREFIX));

            ElementSupport.setDocumentElement(myDocument, otherRoot);
            list = ElementSupport.getChildElements(myDocument);
            Assert.assertEquals(list.size(), 1, "setDocumentElement: One child after element re-set");
            Assert.assertEquals(list.get(0), otherRoot, "setDocumentElement: after element re-set");

        } finally {
            parserPool.returnBuilder(builder);
        }
    }

    private <T> T nullValue() {
        return null;
    }

}