/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.xml;

import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;

import javax.xml.transform.stream.StreamSource;
import javax.xml.validation.Schema;
import javax.xml.validation.Validator;

import org.springframework.core.io.ClassPathResource;
import org.testng.Assert;
import org.testng.annotations.BeforeClass;
import org.testng.annotations.Test;
import org.xml.sax.SAXException;

import net.shibboleth.shared.annotation.constraint.NonnullBeforeTest;
import net.shibboleth.shared.component.ComponentInitializationException;

/**
 * Tests for {@link NamespaceSupport}
 */
@SuppressWarnings("javadoc")
public class SchemaBuilderTest {

    private static final String TEST_DIR = "/net/shibboleth/shared/xml/schemaBuilderTestDir/";

    private static final String FILE_ROOT = "src/test/resources/";

    private static final String FIRST_SCHEMA_FILE = "schemaBuilderTest-schemaFirstLoaded.xsd";

    private static final String SECOND_SCHEMA_FILE = "schemaBuilderTest-schemaSecondLoaded.xsd";

    @NonnullBeforeTest private URL works;

    @NonnullBeforeTest private URL fails;

    private StreamSource workingSource() throws IOException {
        return new StreamSource(works.openStream());
    }

    private StreamSource failingSource() throws IOException {
        return new StreamSource(fails.openStream());
    }

    @BeforeClass public void setup()  {
        works = getClass().getResource(TEST_DIR + "schemaBuilderTest-works.xml");
        fails = getClass().getResource(TEST_DIR + "schemaBuilderTest-fails.xml");
        assert works != null && fails != null;
    }

    @Test public void testFiles() throws SAXException, IOException {
        FileInputStream first = new FileInputStream(FILE_ROOT + TEST_DIR + FIRST_SCHEMA_FILE);
        FileInputStream second = new FileInputStream(FILE_ROOT + TEST_DIR + SECOND_SCHEMA_FILE);

        final SchemaBuilder builder = new SchemaBuilder();
        builder.addSchema(first);
        builder.addSchema(second);
        Schema schema = builder.buildSchema();

        Validator validator = schema.newValidator();

        validator.validate(workingSource());

        boolean thrown = false;
        try {
            validator.validate(failingSource());
        } catch (final Exception e) {
            thrown = true;
        }
        Assert.assertTrue(thrown, "Should fail to validate");
    }

    @Test public void testInputStream() throws SAXException, IOException, ComponentInitializationException {
        
        try (final InputStream first = getClass().getResourceAsStream(TEST_DIR + FIRST_SCHEMA_FILE);
                final InputStream second = getClass().getResourceAsStream(TEST_DIR + SECOND_SCHEMA_FILE)) {

            final SchemaBuilder builder = new SchemaBuilder();
            builder.addSchema(first);
            builder.addSchema(second);
            Schema schema = builder.buildSchema();
    
            Validator validator = schema.newValidator();
    
            validator.validate(workingSource());
    
            boolean thrown = false;
            try {
                validator.validate(failingSource());
            } catch (final Exception e) {
                thrown = true;
            }
            Assert.assertTrue(thrown, "Should fail to validate");
        }
    }

    /**
     * May re-enable if cycle dependency resolved.
     */
    @Test public void testResource() throws SAXException, IOException, ComponentInitializationException {
        net.shibboleth.shared.resource.Resource first = TestResourceConverter.of(new ClassPathResource(TEST_DIR + FIRST_SCHEMA_FILE));
        net.shibboleth.shared.resource.Resource second = TestResourceConverter.of(new ClassPathResource(TEST_DIR + SECOND_SCHEMA_FILE));

        final SchemaBuilder builder = new SchemaBuilder();
        builder.addSchema(first);
        builder.addSchema(second);
        Schema schema = builder.buildSchema();

        Validator validator = schema.newValidator();

        validator.validate(workingSource());

        boolean thrown = false;
        try {
            validator.validate(failingSource());
        } catch (final Exception e) {
            thrown = true;
        }
        Assert.assertTrue(thrown, "Should fail to validate");
    }

}