/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.xml.impl;

import java.io.IOException;
import java.io.InputStream;
import java.io.StringReader;
import java.util.Map;

import javax.xml.XMLConstants;

import org.testng.annotations.BeforeClass;
import org.testng.annotations.Test;
import org.w3c.dom.Document;
import org.w3c.dom.ls.DOMImplementationLS;
import org.xml.sax.SAXException;

import net.shibboleth.shared.annotation.constraint.NonnullBeforeTest;
import net.shibboleth.shared.component.ComponentInitializationException;
import net.shibboleth.shared.xml.XMLParserException;

/**
 * Tests for well-formedness bugs.
 */
public class WellFormedTest {

    @NonnullBeforeTest private BasicParserPool parserPool;

    @BeforeClass public void setup()
            throws XMLParserException, ComponentInitializationException, SAXException, IOException {
        parserPool = new BasicParserPool();
        parserPool.setBuilderFeatures(Map.of(
                XMLConstants.FEATURE_SECURE_PROCESSING, true,
                "http://apache.org/xml/features/nonvalidating/load-external-dtd", false)
                );
        parserPool.initialize();
    }

    @Test(expectedExceptions=XMLParserException.class)
    public void test1NoDTD() throws Exception {
        // This needs to throw an error because xmlns and xmlns: should be seen as duplicated.
        // With no DTD, this does throw, and we want to make sure it does.
        // If this ever squawks, the parser was broken by some regression.

        try (final InputStream is = getClass().getResourceAsStream("trailingColon.xml")) {
            parserPool.parse(is);
        }
    }

    @Test(expectedExceptions=XMLParserException.class)
    public void test1DTD() throws Exception {
        
        // This also needs to throw on the parse and does at the moment.
        // If this ever squawks, the parser was broken by some regression.
        
        try (final InputStream is = getClass().getResourceAsStream("trailingColonDTD.xml")) {
            parserPool.parse(is);
        }
    }

    @Test(expectedExceptions=XMLParserException.class, enabled=false)
    public void test2() throws Exception {

        // This also needs to throw and I think was at one point.
        // Either my notes are bad or this is a regression in the parser around DTDs.
        // We're insulated from DTD bugs in our software in practice, but the core code may be suspect.
        
        final Document doc1;
        try (final InputStream is = getClass().getResourceAsStream("crazyDTD.xml")) {
            doc1 = parserPool.parse(is);
        }
        
        final String s =
                ((DOMImplementationLS) doc1.getImplementation()).createLSSerializer().writeToString(doc1);
        parserPool.parse(new StringReader(s));
    }

}
